<?php

/**
 * Provides the Audit Export Derived tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Derived tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\audit_export_tool\Plugin\tool\Tool\Deriver\AuditExportToolDeriver;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Derived tool for accessing specific audit reports.
 *
 * This tool is not meant to be used directly but serves as the implementation
 * for derived audit-specific tools created by AuditExportToolDeriver.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_derived',
    label: new TranslatableMarkup('Get Audit Report (Derived)'),
    description: new TranslatableMarkup(
        'Base class for derived audit-specific tools. Not used directly.'
    ),
    operation: ToolOperation::Read,
    deriver: AuditExportToolDeriver::class,
    input_definitions: [
    'page' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Page'),
        description: new TranslatableMarkup('Page number for pagination (1-based).'),
        default_value: 1,
        required: false,
        constraints: ['Range' => ['min' => 1]],
    ),
    'limit' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Limit'),
        description: new TranslatableMarkup('Maximum number of rows per page.'),
        default_value: 100,
        required: false,
        constraints: ['Range' => ['min' => 0, 'max' => 1000]],
    ),
    ],
    output_definitions: [
    'audit_id' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The audit machine name queried.'),
    ),
    'headers' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Headers'),
        description: new TranslatableMarkup('Column headers for the audit data.'),
    ),
    'data' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Report Data'),
        description: new TranslatableMarkup('The audit report data rows.'),
    ),
    'total_rows' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Total Rows'),
        description: new TranslatableMarkup('Total rows in the full report.'),
    ),
    'has_more' => new ContextDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Has More'),
        description: new TranslatableMarkup(
            'Whether there are more rows available beyond this response.'
        ),
    ),
    ],
)]
final class AuditExportDerived extends ToolBase
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * Gets the audit ID for this derived tool.
     *
     * @return string
     *   The audit ID extracted from the derivative ID.
     */
    protected function getAuditId(): string
    {
        // The derivative ID is the audit ID with special characters replaced.
        // E.g., 'content_type_audit' from 'audit_export_derived:content_type_audit'.
        $derivative_id = $this->getDerivativeId();
        if ($derivative_id) {
            // The deriver replaces ':', '-', '.' with '_' in the derivative ID,
            // but the original audit_id is used when creating the definition.
            // We need to look up the actual audit plugin to find a match.
            $definitions = $this->auditPluginManager->getDefinitions();
            foreach (array_keys($definitions) as $audit_id) {
                $normalized = str_replace([':', '-', '.'], '_', $audit_id);
                if ($normalized === $derivative_id) {
                    return $audit_id;
                }
            }
            // If no match found by normalization, return the derivative_id as-is
            // in case the audit_id didn't need normalization.
            return $derivative_id;
        }
        return '';
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $this->getAuditId();
        $page = (int) ($values['page'] ?? 1);
        $limit = (int) ($values['limit'] ?? 100);

        if (empty($audit_id)) {
            return ExecutableResult::failure(
                $this->t('No audit ID configured for this derived tool.')
            );
        }

        try {
            // Validate audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Get plugin and headers.
            $plugin = $this->auditPluginManager->createInstance($audit_id);
            $headers = $plugin->getHeaders();

            // Get full report data.
            $all_data = $this->auditReport->getReportData($audit_id);

            if (empty($all_data) || !is_array($all_data)) {
                return ExecutableResult::success(
                    $this->t(
                        'No data available for audit "@id".',
                        ['@id' => $audit_id]
                    ),
                    [
                    'audit_id' => $audit_id,
                    'headers' => $headers,
                    'data' => [],
                    'total_rows' => 0,
                    'has_more' => false,
                    ]
                );
            }

            $total_rows = count($all_data);

            // Calculate pagination.
            if ($limit > 0) {
                $total_pages = (int) ceil($total_rows / $limit);
                $offset = ($page - 1) * $limit;
                $data = array_slice($all_data, $offset, $limit);
                $has_more = $page < $total_pages;
            } else {
                $data = $all_data;
                $has_more = false;
            }

            return ExecutableResult::success(
                $this->t(
                    'Retrieved @count of @total rows for "@id".',
                    [
                    '@count' => count($data),
                    '@total' => $total_rows,
                    '@id' => $audit_id,
                    ]
                ),
                [
                'audit_id' => $audit_id,
                'headers' => $headers,
                'data' => $data,
                'total_rows' => $total_rows,
                'has_more' => $has_more,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to retrieve audit data: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'view audit export reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
