<?php

/**
 * Provides the Audit Export List tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export List tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to list all available audit export plugins.
 *
 * This tool provides information about all registered audit plugins,
 * including their IDs, labels, descriptions, groups, and last run dates.
 * Use this tool to discover what audits are available before running them.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_list',
    label: new TranslatableMarkup('List Audit Export Plugins'),
    description: new TranslatableMarkup(
        'Lists all available audit export plugins with their metadata.
      Use this to discover available audits before running them.'
    ),
    operation: ToolOperation::Explain,
    input_definitions: [
    'group' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Filter by Group'),
        description: new TranslatableMarkup(
            'Filter audits by group name (e.g., "general", "content").'
        ),
        required: false,
    ),
    'include_last_run' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Include Last Run Date'),
        description: new TranslatableMarkup('Include last run timestamp.'),
        default_value: true,
    ),
    ],
    output_definitions: [
    'audits' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Available Audits'),
        description: new TranslatableMarkup('Audit definitions keyed by ID.'),
    ),
    'count' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Audit Count'),
        description: new TranslatableMarkup('Total number of audits returned.'),
    ),
    'groups' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Available Groups'),
        description: new TranslatableMarkup('List of unique audit group names.'),
    ),
    ],
)]
final class AuditExportList extends ToolBase
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $group_filter = !empty($values['group'])
        ? (string) $values['group']
        : null;
        $include_last_run = (bool) ($values['include_last_run'] ?? true);

        try {
            $definitions = $this->auditPluginManager->getDefinitions();
            $audits = [];
            $groups = [];

            foreach ($definitions as $id => $definition) {
                $audit_group = $definition['group'] ?? 'general';
                $groups[$audit_group] = $audit_group;

                // Apply group filter if specified.
                if ($group_filter && $audit_group !== $group_filter) {
                    continue;
                }

                $audit_info = [
                'id' => $id,
                'label' => (string) $definition['label'],
                'description' => (string) ($definition['description'] ?? ''),
                'group' => $audit_group,
                'identifier' => $definition['identifier'] ?? null,
                'data_type' => $definition['data_type'] ?? 'flat',
                'provider' => $definition['provider'] ?? 'unknown',
                'dependencies' => $definition['dependencies'] ?? [],
                ];

                if ($include_last_run) {
                    $last_date = $this->auditReport->getLastProcessedDate($id);
                    $audit_info['last_run'] = $last_date ?: null;
                    $audit_info['has_data'] = $last_date !== null;
                }

                $audits[$id] = $audit_info;
            }

            return ExecutableResult::success(
                $this->t(
                    'Found @count audit plugins.',
                    ['@count' => count($audits)]
                ),
                [
                'audits' => $audits,
                'count' => count($audits),
                'groups' => array_values($groups),
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to list audits: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'view audit export reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
