<?php

/**
 * Provides the Audit Export Queue tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Queue tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Cron\AuditExportCron;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to queue audits for background processing.
 *
 * This tool is preferred for large audits as it uses the queue system
 * rather than running synchronously.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_queue',
    label: new TranslatableMarkup('Queue Audit Export'),
    description: new TranslatableMarkup(
        'Queues one or all audits for background processing.
      Use this for large audits to avoid timeouts.'
    ),
    operation: ToolOperation::Trigger,
    destructive: false,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup(
            'The audit to queue. Leave empty to queue ALL audits.'
        ),
        required: false,
    ),
    'queue_timeout' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Queue Timeout'),
        description: new TranslatableMarkup('Timeout in seconds. Default is 120.'),
        default_value: 120,
        constraints: [
        'Range' => ['min' => 30, 'max' => 600],
        ],
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'queued_count' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Queued Count'),
        description: new TranslatableMarkup('Number of items added to queue.'),
    ),
    'queued_audits' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Queued Audits'),
        description: new TranslatableMarkup('List of audit IDs that were queued.'),
    ),
    'queue_name' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Queue Name'),
        description: new TranslatableMarkup('The name of the queue.'),
    ),
    'queue_size' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Queue Size'),
        description: new TranslatableMarkup('Total items in queue after operation.'),
    ),
    ],
)]
final class AuditExportQueue extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit export cron service.
     *
     * @var \Drupal\audit_export_core\Cron\AuditExportCron
     */
    protected AuditExportCron $auditCron;

    /**
     * The queue factory.
     *
     * @var \Drupal\Core\Queue\QueueFactory
     */
    protected QueueFactory $queueFactory;

    /**
     * The state service.
     *
     * @var \Drupal\Core\State\StateInterface
     */
    protected StateInterface $state;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditCron = $container->get('audit_export_core.cron');
        $instance->queueFactory = $container->get('queue');
        $instance->state = $container->get('state');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_merge([''], array_keys($definitions));
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = !empty($values['audit_id'])
        ? (string) $values['audit_id']
        : null;
        $queue_timeout = (int) ($values['queue_timeout'] ?? 120);

        try {
            $queued_audits = [];
            $queue = $this->queueFactory->get('audit_export_processor');
            $queue_name = 'audit_export_processor';

            if (empty($audit_id)) {
                // Queue all audits.
                $count = $this->auditCron->queueAudits($queue_timeout);
                $queued_audits = array_keys(
                    $this->auditPluginManager->getDefinitions()
                );
            } else {
                // Validate audit exists.
                if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                    return ExecutableResult::failure(
                        $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                    );
                }

                // Set queue timeout in state.
                $this->state->set('audit_export.queue_timeout', $queue_timeout);

                // Queue single audit.
                $definition = $this->auditPluginManager->getDefinition($audit_id);
                $plugin = $this->auditPluginManager->createInstance($audit_id);
                $data_to_process = $plugin->prepareData();

                if (empty($data_to_process)) {
                    // Queue placeholder for empty data.
                    $queue->createItem(
                        [
                        'plugin_id' => $audit_id,
                        'group' => $definition['group'] ?? 'general',
                        'empty_data' => true,
                        ]
                    );
                    $count = 1;
                } else {
                    // Split into batches.
                    $estimated_items_per_minute = 300;
                    $items_per_second = $estimated_items_per_minute / 60;
                    $items_per_timeout = max(
                        10,
                        min(200, ceil($items_per_second * $queue_timeout))
                    );
                    $chunk_size = (int) $items_per_timeout;
                    $batches = array_chunk($data_to_process, $chunk_size);
                    $count = 0;

                    foreach ($batches as $index => $batch) {
                            $queue->createItem(
                                [
                                'plugin_id' => $audit_id,
                                'group' => $definition['group'] ?? 'general',
                                'batch_data' => $batch,
                                'batch_index' => $index,
                                'total_batches' => count($batches),
                                'continued_processing' => $index > 0,
                                ]
                            );
                            $count++;
                    }
                }

                $queued_audits = [$audit_id];
            }

            $queue_size = $queue->numberOfItems();

            return ExecutableResult::success(
                $this->t(
                    'Queued @count items for processing.',
                    ['@count' => $count]
                ),
                [
                'queued_count' => $count,
                'queued_audits' => $queued_audits,
                'queue_name' => $queue_name,
                'queue_size' => $queue_size,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to queue audits: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission($account, 'run audit export');
        return $return_as_object ? $access : $access->isAllowed();
    }

}
