<?php

/**
 * Provides the Audit Export Report Get tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Report Get tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to retrieve audit report data.
 *
 * Returns the stored audit report data with optional pagination and filtering.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_report_get',
    label: new TranslatableMarkup('Get Audit Report'),
    description: new TranslatableMarkup(
        'Retrieves stored audit report data for a specific audit.
      Supports pagination for large reports.'
    ),
    operation: ToolOperation::Read,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The machine name of the audit.'),
        required: true,
    ),
    'page' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Page'),
        description: new TranslatableMarkup('Page number (1-based). Default is 1.'),
        default_value: 1,
        required: false,
        constraints: [
        'Range' => ['min' => 1],
        ],
    ),
    'limit' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Limit'),
        description: new TranslatableMarkup('Max rows per page. 0 for all.'),
        default_value: 100,
        required: false,
        constraints: [
        'Range' => ['min' => 0, 'max' => 1000],
        ],
    ),
    'include_headers' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Include Headers'),
        description: new TranslatableMarkup('Whether to include column headers.'),
        default_value: true,
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'audit_id' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The ID of the audit report.'),
    ),
    'headers' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Headers'),
        description: new TranslatableMarkup('Column headers for the audit data.'),
    ),
    'data' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Report Data'),
        description: new TranslatableMarkup('The audit report data rows.'),
    ),
    'total_rows' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Total Rows'),
        description: new TranslatableMarkup('Total rows in the full report.'),
    ),
    'returned_rows' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Returned Rows'),
        description: new TranslatableMarkup('Number of rows returned.'),
    ),
    'current_page' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Current Page'),
        description: new TranslatableMarkup('The current page number.'),
    ),
    'total_pages' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Total Pages'),
        description: new TranslatableMarkup('Total number of pages available.'),
    ),
    'has_more' => new ContextDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Has More'),
        description: new TranslatableMarkup('Whether more rows are available.'),
    ),
    'last_updated' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Last Updated'),
        description: new TranslatableMarkup('Timestamp of last generation.'),
    ),
    ],
)]
final class AuditExportReportGet extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_keys($definitions);
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $values['audit_id'];
        $page = (int) ($values['page'] ?? 1);
        $limit = (int) ($values['limit'] ?? 100);
        $include_headers = (bool) ($values['include_headers'] ?? true);

        try {
            // Validate audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Get full report data.
            $all_data = $this->auditReport->getReportData($audit_id);

            if (empty($all_data) || !is_array($all_data)) {
                return ExecutableResult::success(
                    $this->t(
                        'No report data for audit "@id". Run the audit first.',
                        ['@id' => $audit_id]
                    ),
                    [
                    'audit_id' => $audit_id,
                    'headers' => [],
                    'data' => [],
                    'total_rows' => 0,
                    'returned_rows' => 0,
                    'current_page' => 1,
                    'total_pages' => 0,
                    'has_more' => false,
                    'last_updated' => null,
                    ]
                );
            }

            $total_rows = count($all_data);

            // Calculate pagination.
            if ($limit > 0) {
                $total_pages = (int) ceil($total_rows / $limit);
                $offset = ($page - 1) * $limit;
                $data = array_slice($all_data, $offset, $limit);
            } else {
                $total_pages = 1;
                $data = $all_data;
            }

            $returned_rows = count($data);
            $has_more = $page < $total_pages;

            // Get headers.
            $headers = [];
            if ($include_headers) {
                $plugin = $this->auditPluginManager->createInstance($audit_id);
                $headers = $plugin->getHeaders();
            }

            // Get last updated timestamp.
            $last_updated = $this->auditReport->getLastProcessedDate($audit_id);

            return ExecutableResult::success(
                $this->t(
                    'Retrieved @count of @total rows for "@id" (page @page/@pages).',
                    [
                    '@count' => $returned_rows,
                    '@total' => $total_rows,
                    '@id' => $audit_id,
                    '@page' => $page,
                    '@pages' => $total_pages,
                    ]
                ),
                [
                'audit_id' => $audit_id,
                'headers' => $headers,
                'data' => $data,
                'total_rows' => $total_rows,
                'returned_rows' => $returned_rows,
                'current_page' => $page,
                'total_pages' => $total_pages,
                'has_more' => $has_more,
                'last_updated' => $last_updated,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to retrieve report for "@id": @error',
                    [
                    '@id' => $audit_id,
                    '@error' => $e->getMessage(),
                    ]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'view audit export reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
