<?php

/**
 * Provides the Audit Export Report Row tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Report Row tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to retrieve a specific row from an audit report.
 *
 * Allows fetching individual rows by index or by searching for a value
 * in the identifier column.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_report_row',
    label: new TranslatableMarkup('Get Audit Report Row'),
    description: new TranslatableMarkup(
        'Retrieves a specific row from an audit report by row index or
      by searching for an identifier value.'
    ),
    operation: ToolOperation::Read,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The machine name of the audit.'),
        required: true,
    ),
    'row_index' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Row Index'),
        description: new TranslatableMarkup(
            'The zero-based index of the row to retrieve.
          Use this OR identifier_value, not both.'
        ),
        required: false,
        constraints: [
        'Range' => ['min' => 0],
        ],
    ),
    'identifier_value' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Identifier Value'),
        description: new TranslatableMarkup(
            'Search for a row where the identifier column matches this value.
          Use this OR row_index, not both.'
        ),
        required: false,
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'found' => new ContextDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Found'),
        description: new TranslatableMarkup('Whether a matching row was found.'),
    ),
    'row_index' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Row Index'),
        description: new TranslatableMarkup('The index of the returned row.'),
    ),
    'headers' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Headers'),
        description: new TranslatableMarkup('Column headers for context.'),
    ),
    'row' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Row Data'),
        description: new TranslatableMarkup('The row data as an array.'),
    ),
    'row_keyed' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Row Data (Keyed)'),
        description: new TranslatableMarkup(
            'The row data as a key-value map using headers as keys.'
        ),
    ),
    ],
)]
final class AuditExportReportRow extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_keys($definitions);
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $values['audit_id'];
        // Check if row_index was explicitly provided (not just defaulted to NULL).
        // We need to handle the case where row_index could be 0, which is valid.
        $row_index_provided = array_key_exists('row_index', $values)
        && $values['row_index'] !== null
        && $values['row_index'] !== '';
        $row_index = $row_index_provided ? (int) $values['row_index'] : null;
        // identifier_value can be '0' which is valid, so use isset check.
        $identifier_value = (isset($values['identifier_value'])
        && $values['identifier_value'] !== '')
        ? (string) $values['identifier_value']
        : null;

        // Validate that at least one lookup method is provided.
        if ($row_index === null && $identifier_value === null) {
            return ExecutableResult::failure(
                $this->t('You must provide either row_index or identifier_value.')
            );
        }

        try {
            // Validate audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Get plugin and data.
            $plugin = $this->auditPluginManager->createInstance($audit_id);
            // Convert headers to strings, handling TranslatableMarkup, null, etc.
            $headers = array_map(
                function ($header) {
                    if ($header === null) {
                        return '';
                    }
                    if (is_object($header) && method_exists($header, '__toString')) {
                        return (string) $header;
                    }
                    if (is_scalar($header)) {
                        return (string) $header;
                    }
                    // For arrays or other types, return empty string.
                    return '';
                },
                $plugin->getHeaders()
            );
            $plugin_definition = $this->auditPluginManager->getDefinition($audit_id);
            $identifier_column = $plugin_definition['identifier'] ?? null;

            $all_data = $this->auditReport->getReportData($audit_id);

            if (empty($all_data) || !is_array($all_data)) {
                return ExecutableResult::failure(
                    $this->t(
                        'No report data available for audit "@id".',
                        ['@id' => $audit_id]
                    )
                );
            }

            $found_row = null;
            $found_index = null;

            // Lookup by index.
            if ($row_index !== null) {
                if (isset($all_data[$row_index])) {
                    $found_row = $all_data[$row_index];
                    $found_index = $row_index;
                }
            } elseif ($identifier_value !== null) {
                // Lookup by identifier value.
                // Determine identifier column index. The identifier from the plugin
                // definition (e.g., "machine_name") may not match the display header
                // exactly (e.g., "Machine Name"), so we try multiple strategies.
                $identifier_col_index = null;
                if ($identifier_column) {
                    // Normalize identifier for comparison.
                    $normalized_identifier = strtolower(
                        str_replace(['_', ' ', '-'], '', $identifier_column)
                    );

                    foreach ($headers as $idx => $header) {
                        // Ensure header is a string for comparison.
                        $header_str = is_string($header) ? $header : '';

                        // Exact match.
                        if ($header_str === $identifier_column) {
                            $identifier_col_index = $idx;
                            break;
                        }

                        // Skip empty headers for normalized matching.
                        if ($header_str === '') {
                            continue;
                        }

                        // Normalized match.
                        $normalized_header = strtolower(
                            str_replace(['_', ' ', '-'], '', $header_str)
                        );
                        if ($normalized_header === $normalized_identifier) {
                            $identifier_col_index = $idx;
                            break;
                        }

                        // Header contains identifier as substring.
                        if (str_contains(
                            $normalized_header,
                            $normalized_identifier
                        )
                        ) {
                            $identifier_col_index = $idx;
                            break;
                        }
                    }
                }

                // Default to first column if identifier not found.
                if ($identifier_col_index === null) {
                    $identifier_col_index = 0;
                }

                // Search for matching row using multiple strategies:
                // 1. Exact match
                // 2. Machine name match for "Label (machine_name)" format
                // 3. Case-insensitive match
                $search_value = (string) $identifier_value;
                $search_value_lower = strtolower($search_value);

                foreach ($all_data as $idx => $row) {
                    // Skip non-array rows.
                    if (!is_array($row)) {
                        continue;
                    }
                    $cell_value = $row[$identifier_col_index] ?? null;
                    // Skip non-scalar cell values that can't be compared as strings.
                    if ($cell_value === null || (!is_scalar($cell_value)
                        && !(is_object($cell_value)
                        && method_exists($cell_value, '__toString')))
                    ) {
                        continue;
                    }
                    $cell_str = (string) $cell_value;

                    // Strategy 1: Exact match.
                    if ($cell_str === $search_value) {
                        $found_row = $row;
                        $found_index = $idx;
                        break;
                    }

                    // Strategy 2: Machine name match for "Label (machine_name)" format.
                    // Extract the value inside parentheses and compare.
                    if (preg_match('/\(([^)]+)\)\s*$/', $cell_str, $matches)) {
                        if ($matches[1] === $search_value
                            || strtolower($matches[1]) === $search_value_lower
                        ) {
                            $found_row = $row;
                            $found_index = $idx;
                            break;
                        }
                    }

                    // Strategy 3: Case-insensitive exact match.
                    if (strtolower($cell_str) === $search_value_lower) {
                        $found_row = $row;
                        $found_index = $idx;
                        break;
                    }
                }
            }

            if ($found_row === null) {
                return ExecutableResult::success(
                    $this->t('No matching row found.'),
                    [
                    'found' => false,
                    'row_index' => null,
                    'headers' => $headers,
                    'row' => [],
                    'row_keyed' => [],
                    ]
                );
            }

            // Build keyed version (headers are already converted to strings above).
            $row_keyed = [];
            foreach ($headers as $idx => $header) {
                $row_keyed[$header] = $found_row[$idx] ?? null;
            }

            return ExecutableResult::success(
                $this->t('Found row at index @index.', ['@index' => $found_index]),
                [
                'found' => true,
                'row_index' => $found_index,
                'headers' => $headers,
                'row' => $found_row,
                'row_keyed' => $row_keyed,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to retrieve row: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'view audit export reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
