<?php

/**
 * Provides the Audit Export Run tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Run tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to execute a specific audit and generate its report.
 *
 * This tool runs an audit synchronously and returns the generated data.
 * For large audits, consider using audit_export_queue instead.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_run',
    label: new TranslatableMarkup('Run Audit'),
    description: new TranslatableMarkup(
        'Executes a specific audit plugin and generates/updates its report data.
      Returns the processed audit data upon completion.'
    ),
    operation: ToolOperation::Trigger,
    destructive: false,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The machine name of the audit to run.'),
        required: true,
    ),
    'clear_existing' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Clear Existing Data'),
        description: new TranslatableMarkup('Whether to clear existing data first.'),
        default_value: true,
    ),
    'return_data' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Return Data'),
        description: new TranslatableMarkup(
            'Whether to return full data. Set FALSE for large audits.'
        ),
        default_value: true,
    ),
    'limit' => new InputDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Row Limit'),
        description: new TranslatableMarkup('Max rows to process. 0 for no limit.'),
        default_value: 0,
        required: false,
        constraints: [
        'Range' => ['min' => 0, 'max' => 10000],
        ],
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'success' => new ContextDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Success'),
        description: new TranslatableMarkup('Whether the audit completed.'),
    ),
    'audit_id' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The ID of the audit that was run.'),
    ),
    'row_count' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Row Count'),
        description: new TranslatableMarkup('Number of rows generated.'),
    ),
    'headers' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Headers'),
        description: new TranslatableMarkup('Column headers for the audit data.'),
    ),
    'data' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Audit Data'),
        description: new TranslatableMarkup('The processed audit data rows.'),
    ),
    'execution_time' => new ContextDefinition(
        data_type: 'float',
        label: new TranslatableMarkup('Execution Time'),
        description: new TranslatableMarkup('Time taken in seconds.'),
    ),
    ],
)]
final class AuditExportRun extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_keys($definitions);
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $values['audit_id'];
        $return_data = (bool) ($values['return_data'] ?? true);
        $limit = (int) ($values['limit'] ?? 0);

        $start_time = microtime(true);

        try {
            // Validate audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Clear existing data before running to prevent duplication.
            // Even with clear_existing=FALSE, we still clear to avoid appending
            // duplicate rows. The parameter name is kept for backwards compatibility
            // but the behavior is to always start fresh when running an audit.
            $this->auditReport->clearReportData($audit_id);

            // Create plugin instance.
            $plugin = $this->auditPluginManager->createInstance($audit_id);
            $headers = $plugin->getHeaders();

            // Prepare data.
            $items = $plugin->prepareData();

            // Apply limit if specified.
            if ($limit > 0 && count($items) > $limit) {
                $items = array_slice($items, 0, $limit);
            }

            // Process each item.
            $processed_data = [];
            foreach ($items as $item) {
                $row = $plugin->processData(['row_data' => $item]);
                $processed_data[] = $row;

                // Append to report.
                $this->auditReport->appendReportData($audit_id, $row);
            }

            // Update report timestamp.
            $this->auditReport->updateReportDate($audit_id);

            $execution_time = round(microtime(true) - $start_time, 3);

            $output = [
            'success' => true,
            'audit_id' => $audit_id,
            'row_count' => count($processed_data),
            'headers' => $headers,
            'execution_time' => $execution_time,
            ];

            // Include data only if requested.
            if ($return_data) {
                $output['data'] = $processed_data;
            } else {
                $output['data'] = [];
            }

            return ExecutableResult::success(
                $this->t(
                    'Audit "@id" completed with @count rows in @time seconds.',
                    [
                    '@id' => $audit_id,
                    '@count' => count($processed_data),
                    '@time' => $execution_time,
                    ]
                ),
                $output
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Audit "@id" failed: @error',
                    [
                    '@id' => $audit_id,
                    '@error' => $e->getMessage(),
                    ]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission($account, 'run audit export');
        return $return_as_object ? $access : $access->isAllowed();
    }

}
