<?php

/**
 * Provides the Audit Export Tool Deriver class.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Tool Deriver class.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool\Deriver;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_tool\Plugin\tool\Tool\AuditExportDerived;
use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Tool\ToolDefinition;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Derives audit-specific tool plugins from registered audit export plugins.
 *
 * This deriver creates a "Get [Audit Name]" tool for each registered
 * audit export plugin, allowing for more intuitive tool discovery.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
class AuditExportToolDeriver extends DeriverBase implements ContainerDeriverInterface
{

    use StringTranslationTrait;

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container      Service container.
     * @param string             $base_plugin_id Base plugin ID.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        $base_plugin_id,
    ): static {
        $instance = new static();
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param mixed $base_plugin_definition Base plugin definition.
     *
     * @return array
     *   The derivative definitions.
     */
    public function getDerivativeDefinitions($base_plugin_definition): array
    {
        if (!empty($this->derivatives)) {
            return $this->derivatives;
        }

        $definitions = $this->auditPluginManager->getDefinitions();

        foreach ($definitions as $audit_id => $audit_definition) {
            $audit_label = (string) $audit_definition['label'];
            $desc = $audit_definition['description'] ?? $audit_label;
            $audit_description = (string) $desc;

            // Create derivative ID using the audit ID.
            $derivative_id = str_replace([':', '-', '.'], '_', $audit_id);

            // Build the derivative definition.
            $derivative = [
            'id' => 'audit_export_derived:' . $derivative_id,
            'label' => new TranslatableMarkup(
                'Get @audit Report',
                ['@audit' => $audit_label]
            ),
            'description' => new TranslatableMarkup(
                'Retrieves the @audit audit report data with pagination. @desc',
                ['@audit' => $audit_label, '@desc' => $audit_description]
            ),
            'operation' => ToolOperation::Read,
            'destructive' => false,
            'provider' => 'audit_export_tool',
            'audit_id' => $audit_id,
            'class' => AuditExportDerived::class,
            'input_definitions' => [
            'page' => new InputDefinition(
                data_type: 'integer',
                label: new TranslatableMarkup('Page'),
                description: new TranslatableMarkup('Page number (1-based).'),
                default_value: 1,
                required: false,
                constraints: ['Range' => ['min' => 1]],
            ),
            'limit' => new InputDefinition(
                data_type: 'integer',
                label: new TranslatableMarkup('Limit'),
                description: new TranslatableMarkup('Max rows per page. 0 for all.'),
                default_value: 100,
                required: false,
                constraints: ['Range' => ['min' => 0, 'max' => 1000]],
            ),
            ],
            'output_definitions' => [
            'audit_id' => new ContextDefinition(
                data_type: 'string',
                label: new TranslatableMarkup('Audit ID'),
            ),
            'headers' => new ContextDefinition(
                data_type: 'any',
                label: new TranslatableMarkup('Headers'),
            ),
            'data' => new ContextDefinition(
                data_type: 'any',
                label: new TranslatableMarkup('Report Data'),
            ),
            'total_rows' => new ContextDefinition(
                data_type: 'integer',
                label: new TranslatableMarkup('Total Rows'),
            ),
            'has_more' => new ContextDefinition(
                data_type: 'boolean',
                label: new TranslatableMarkup('Has More'),
            ),
            ],
            ];

            $this->derivatives[$derivative_id] = new ToolDefinition($derivative);
        }

        return $this->derivatives;
    }

}
