<?php

/**
 * Auditing all site blocks enabled and placed on the site in various ways.
 */
class AuditExportBlocksEnabled extends AuditExportAuditData {

  /**
   * Build headers for Blocks Enabled report.
   */
  public function __construct() {
    $this->setHeaders(['Block', 'Module', 'Placement', 'Region', 'Visibility']);
  }

  /**
   * Return all block ids for processing.
   *
   * @return array
   */
  public function prepareData(): array {
    return $this->fetchAllBlocks();
  }

  /**
   * Process all block ids.
   *
   * @param array $params
   *
   * @return array
   */
  public function processData(array $params = []): array {
    return $this->getBlockData($params["row_data"]);
  }

  private function fetchAllBlocks(): array {
    $common_blocks = [];
    foreach ($this->fetchEnabledBlockIds() as $common_block) {
      $skip = FALSE;
      $block_info = $this->fetchBlockInfo($common_block);
      if ($block_info && $this->isReferenceThemeEnabled($block_info)) {
        if ($this->checkBaseTheme($block_info)) {
          $skip = TRUE;
        }

        if ($block_info['module'] == 'views') {
          $skip = $this->processViewsBlock($block_info);
          if ($skip) {
            $skip = TRUE;
          }
        }

        if ($skip == FALSE) {
          $common_blocks[] = $common_block;
        }

      }
    }
    $context_blocks = $this->fetchContextBlocks();

    $merged_blocks = [];

    if (!empty($common_blocks)) {
      foreach ($common_blocks as $common_block) {
        $merged_blocks[] = [
          'type' => 'common',
          'block' => $common_block,
        ];
      }
    }

    if (!empty($context_blocks)) {
      foreach ($context_blocks as $context_block) {
        $merged_blocks[] = [
          'type' => 'context',
          'block' => $context_block,
        ];
      }
    }
    return $merged_blocks;
  }


  private function getBlockData($block) {

    $block_data = [
      'block' => NULL,
      'menu' => NULL,
      'placement' => NULL,
      'region' => NULL,
      'visibility' => NULL
    ];

    if ($block["type"] == 'common') {

      $bid = $block["block"];
      $block_info = $this->fetchBlockInfo($bid);

      if ($block_info && $this->isReferenceThemeEnabled($block_info)) {

        // Omit blocks from base theme. TODO: Add this as a setting?
        if ($this->checkBaseTheme($block_info)) {
          return [];
        }

        // Omit phantom blocks that have been added by past views.
        if ($block_info['module'] == 'views') {
          $skip = $this->processViewsBlock($block_info);
          if ($skip) {
            return [];
          }
        }

        $theme_info = $this->getThemeInfo($block_info['theme']);
        $module_info = $this->getModuleInfo($block_info['module']);
        $block_title = $this->getBlockTitle($block_info, $bid) . " (bid: $bid)";
        $pages = $this->cleanArrayChars($block_info['pages']);
        $visibility = $this->getBlockVisibility($block_info['visibility'], $pages);

        return [
          $block_title, "{$module_info['name']} ({$block_info['module']})",
          "{$theme_info['name']} ({$block_info['theme']})",
          $this->getRegionName($theme_info, $block_info['region']),
          "$visibility $pages"
        ];
      }
    }

    if ($block["type"] == 'context' && module_exists('context')) {

      $block_seg = explode('-', $block["block"]);
      $context_name = $block_seg[0];
      $context = context_load($context_name);

      if (count($block_seg) == 3) {
        $block_source = 'module';
      } elseif (count($block_seg) == 2) {
        $block_source = 'custom';
      } else {
        $block_source = 'undefined';
      }

      if ($block_source == 'module') {
        $context_block = $context->reactions["block"]["blocks"][$block_seg[1] . '-' . $block_seg[2]];
        $block_db = block_load($block_seg[1], $block_seg[2]);
        $block_info = $this->fetchBlockInfo($block_db->bid);
        $block_title = $this->getBlockTitle($block_info, $block_db->bid) . " (bid: $block_db->bid)";
        $theme_info = $this->getThemeInfo(variable_get('theme_default'));

        $block_data = [
          'block' => $block_title,
          'module' => $this->getModuleInfo($block_db->module),
          'placement' => "Context: $context->name",
          'region' => $this->getRegionName($theme_info, $context_block["region"]),
          'visibility' => $this->getContextBlockVisibility($context),
        ];

        return [
          $block_data["block"],
          $block_data["module"]["name"] .  " (" . $block_db->module . ")",
          $block_data["placement"],
          $block_data["region"],
          $block_data["visibility"],
        ];

      }

    }

    return [];
  }

  /**
   * Return all block ids that are enabled.
   *
   * @return array
   */
  private function fetchEnabledBlockIds(): array {
    return db_query("SELECT bid FROM {block} WHERE status = :status", [':status' => 1])->fetchCol();
  }

  private function fetchContextBlocks(): array {
    if (module_exists('context')) {
      $context_blocks = [];
      foreach (context_load() as $context) {
        if (!$context->disabled) {
          if (!empty($context->reactions["block"])) {
            if (!empty($context->reactions["block"]["blocks"])) {
              foreach ($context->reactions["block"]["blocks"] as $key => $context_block) {
                $context_blocks[] = "$context->name-$key";
              }
            }
          }
        }
      }
      return $context_blocks;

    }
    return [];
  }

  /**
   * Query database for more block information.
   *
   * @param $bid
   *
   * @return array
   */
  private function fetchBlockInfo($bid): array {
    return db_query("SELECT theme, status, module, delta, region, visibility, pages
                    FROM {block}
                    WHERE bid = :bid", [':bid' => $bid])->fetchAssoc();
  }

  /**
   * Check if theme is enabled
   *
   * @param $block_info
   *
   * @return bool
   */
  private function isReferenceThemeEnabled($block_info): bool {
    return $block_info['status'] && $this->isThemeEnabled($block_info['theme']);
  }

  /**
   * Check if theme is a base theme of the default theme.
   *
   * @param $block_info
   *
   * @return bool
   */
  private function checkBaseTheme($block_info): bool {
    $default_theme_info = system_get_info('theme', variable_get('theme_default'));
    return (isset($default_theme_info['base theme']) && $default_theme_info['base theme'] == $block_info["theme"]);
  }

  /**
   * @param $theme
   *
   * @return array
   */
  private function getThemeInfo($theme): array {
    return system_get_info('theme', $theme);
  }

  /**
   * @param $module
   *
   * @return array
   */
  private function getModuleInfo($module): array {
    return system_get_info('module', $module);
  }

  /**
   * @param $block_info
   * @param $bid
   *
   * @return string
   */
  private function getBlockTitle($block_info, $bid): string {
    if ($block_info['module'] == 'views') {
      return $this->getViewBlockTitle($block_info, $bid);
    }
    return $this->getBlockDefaultTitle($block_info);
  }

  /**
   * Get visibility info for a blocks managed by core block management.
   *
   * @param $visibility
   *
   * @param $pages
   *
   * @return string
   */
  private function getBlockVisibility($visibility, $pages): string {
    $all_pages_string = NULL;

    // Clean up the output a bit.
    if (empty($pages) && $visibility == BLOCK_VISIBILITY_NOTLISTED) {
      $all_pages_string = t('All pages');
    } elseif ($visibility == BLOCK_VISIBILITY_NOTLISTED) {
      $all_pages_string = t('All pages except:');
    }

    return $visibility == BLOCK_VISIBILITY_NOTLISTED ?
      $all_pages_string :
      t('Only on pages:');
  }

  /**
   * Return conditions and values for contexts.
   *
   * @param $context
   *
   * @return string
   */
  private function getContextBlockVisibility($context): string {
    $visibility_options = [];
    $visibility = NULL;

    // Loop through all context and build accessible array.
    foreach ($context->conditions as $condition_type => $condition) {
      $condition_info = context_get_plugin('condition', $condition_type);
      $visibility_options[$condition_type] = [
        'label' => $condition_info->title,
        'values' => array_values($context->conditions[$condition_type]["values"])
      ];

      // Override sitewide context value to be human-readable.
      if ($condition_type == 'sitewide') {
        if ($visibility_options[$condition_type]['values'][0] == 1) {
          $visibility_options[$condition_type]['values'] = ['Visible on all pages'];
        } else {
          $visibility_options[$condition_type]['values'] = ['Disabled']; // todo: should this just return instead?
        }
      }
    }

    // Break each option onto a new line.
    foreach ($visibility_options as $visibility_option) {
      if (!empty($visibility)) {
        $visibility .= nl2br("\n");
      }
      $visibility .= $visibility_option['label'] . ': ' . implode(', ', $visibility_option['values']);
    }

    return $visibility;
  }

  /**
   * Get the region name where the block is enabled.
   *
   * @param $theme_info
   * @param $region
   *
   * @return string
   */
  private function getRegionName($theme_info, $region): string {
    return isset($theme_info['regions'][$region]) ?
      "{$theme_info['regions'][$region]} ({$region})" :
      t('Undefined region');
  }

  /**
   * @param $theme
   *
   * @return bool
   */
  private function isThemeEnabled($theme): bool {
    $info = system_list('theme');
    return isset($info[$theme]) && $info[$theme]->status;
  }

  /**
   * Return the name of the view for the block title, if a custom title has not
   * been provided or if the title isn't hidden (set to <none>).
   *
   * @param $block_info
   * @param $bid
   *
   * @return string
   */
  private function getViewBlockTitle($block_info, $bid): string {
    return $this->getBlockDefaultTitle($block_info); // Fallback to default title.
  }

  /**
   * Retrieves the default title for a block based on its module and delta.
   *
   * @param array $block_info
   *   An associative array containing information about the block, including
   *   'module' and 'delta' keys.
   *
   * @return string
   *   The default title of the block.
   */
  private function getBlockDefaultTitle(array $block_info): string {
    // Assuming $block_info includes 'module' and 'delta' keys.
    $module = $block_info['module'];
    $delta = $block_info['delta'];

    // Invoke the hook_block_info() to get block definitions from the module.
    $block_definitions = module_invoke($module, 'block_info');

    // Check if the block is defined and has an 'info' attribute.
    if (isset($block_definitions[$delta]) && !empty($block_definitions[$delta]['info'])) {
      // The 'info' attribute typically holds the default/administrative title of the block.
      return t($block_definitions[$delta]['info']);
    }

    // Fallback title if not specified.
    return t('Untitled block');
  }

  /**
   * Special processing for blocks created by the Views module.
   *
   * @param array $block_info
   *   Block information array.
   *
   * @return bool
   *   TRUE if the block should be skipped, FALSE otherwise.
   */
  private function processViewsBlock(array $block_info): bool {
    $view_name = explode('-', $block_info["delta"])[0];
    $view = views_get_view($view_name);

    // Skip if the view is disabled or cannot be loaded.
    if (!$view || $view->disabled) {
      return true;
    }

    // Extract display id from delta and check if it exists in the view.
    if (str_contains($block_info["delta"], '-')) {
      $display_id = explode('-', $block_info["delta"])[1];
      if (!isset($view->display[$display_id])) {
        // Skip if the display does not exist in the view.
        return true;
      }
    }

    return false;
  }

  /**
   * Clean up special characters and return an array.
   *
   * @param string $string
   *
   * @return string
   */
  private function cleanArrayChars(string $string): string {
    // Normalize line endings and explode the string into an array.
    $output = '';
    $normalized_string = str_replace(array("\r\n", "\r"), "\n", $string);
    $array = explode("\n", $normalized_string);

    if (!empty($array[0])) {
      $output = implode(', ', $array);
    }

    return $output;
  }
}
