<?php

/**
 * EntityFieldAudit class to gather field information for all entity types and bundles.
 */
class AuditExportEntityField extends AuditExportAuditData {

  /**
   * Constructor to set up the audit headers.
   */
  public function __construct() {
    $this->setHeaders([
      'Entity',
      'Bundle',
      'Field',
      'Field Type',
      'Module',
      'Referenced Entity',
      'Referenced Bundle(s)',
      'Cardinality',
    ]);
  }

  /**
   * Prepares the initial data set for processing.
   *
   * @return array
   *   An array of arrays, each representing an entity type, bundle, and field instance.
   */
  public function prepareData(): array {
    $entity_info = entity_get_info();
    $entities = [];

    foreach (array_keys($entity_info) as $entity_type) {
      $bundles = (isset($entity_info[$entity_type]['bundles'])) ? array_keys($entity_info[$entity_type]['bundles']) : [$entity_type];

      foreach ($bundles as $bundle) {
        // First, add default properties for the entity type
        $default_properties = $this->getDefaultProperties($entity_type);
        foreach ($default_properties as $property) {
          $entities[] = [
            'entity_type' => $entity_type,
            'bundle' => $bundle,
            'field_name' => $property
          ];
        }

        // Next, add fields provided by the Field API
        $instances = field_info_instances($entity_type, $bundle);
        foreach ($instances as $field_name => $instance) {
          $entities[] = [
            'entity_type' => $entity_type,
            'bundle' => $bundle,
            'field_name' => $field_name
          ];
        }
      }
    }

    return $entities;
  }

  /**
   * Provides a list of default properties for entities that may not use the Field API.
   *
   * @param string $entity_type
   *   The type of entity to get properties for.
   *
   * @return array
   *   An array of property names.
   */
  protected function getDefaultProperties($entity_type) {
    switch ($entity_type) {
      case 'taxonomy_term':
        return ['name', 'description'];
      case 'node':
        return ['title'];
      default:
        return [];
    }
  }

  /**
   * Processes the data for a single entity, bundle, and field.
   *
   * @param array $params
   *   Parameters including 'entity_type', 'bundle', and 'field_name'.
   *
   * @return array
   *   An array of data ready to be exported, including reference field configurations.
   */
  public function processData(array $params = []): array {
    $entity_type = $params['row_data']['entity_type'];
    $bundle = $params['row_data']['bundle'];
    $field_name = $params['row_data']['field_name'];

    // Determine if the field_name is a default property or a field
    $default_properties = $this->getDefaultProperties($entity_type);
    if (in_array($field_name, $default_properties)) {
      $field_type = 'property';
      $module = 'core';
      $referenced_entity_type = '';
      $referenced_bundles = '';
      $cardinality = 'Single';
    } else {
      $field = field_info_field($field_name);
      $field_type = $field['type'];
      $module = $field["module"];

      // Check for reference fields and get configuration
      $reference_info = $this->getReferenceEntities($entity_type, $bundle, $field);
      $referenced_entity_type = $reference_info['entity_type'];
      $referenced_bundles = $reference_info['bundles'];
      $cardinality = isset($field['cardinality']) ? $field['cardinality'] : '';
      $cardinality = ($cardinality == '1') ? 'Single' :  $cardinality;
      $cardinality = $cardinality == '-1' ? 'Unlimited' : $cardinality;
    }

    return [
      $entity_type,
      $bundle,
      $field_name,
      $field_type,
      $module,
      $referenced_entity_type,
      strtolower($referenced_bundles),
      $cardinality
    ];
  }

  /**
   * Helper function to get referenced entity types for reference fields.
   *
   * @param array $field
   *   The field definition array.
   *
   * @return array
   *   A string representation of all referenced entities or bundles.
   */
  protected function getReferenceEntities(string $entity_type, string $bundle, array $field) {
    $info = [
      'entity_type' => '',
      'bundles' => ''
    ];

    switch ($field['type']) {
      case 'entityreference':
        $info['entity_type'] = $field["settings"]["target_type"];
        if (isset($field['settings']['handler_settings']['target_bundles'])) {
          $info['bundles'] = implode(', ', array_keys($field['settings']['handler_settings']['target_bundles']));
        }
        break;

      case 'taxonomy_term_reference':
        // This field type usually references taxonomy terms
        $info['entity_type'] = 'taxonomy_term';
        if (isset($field['settings']['allowed_values']) && is_array($field['settings']['allowed_values'])) {
          $vocabularies = array_map(function($value) {
            return $value['vocabulary'];
          }, $field['settings']['allowed_values']);
          $vocab_names = array_map(function($vocabulary) {
            $vocab = taxonomy_vocabulary_machine_name_load($vocabulary);
            return $vocab ? $vocab->name : 'Unknown vocabulary (' . $vocabulary . ')';
          }, $vocabularies);
          $info['bundles'] = implode(', ', $vocab_names);
        }
        break;

      // This field type references field collections
      case 'field_collection':
        $info['entity_type'] = 'field_collection_item';
        $info['bundles'] = $field['field_name'];
        break;

      // This field type references paragraphs
      case 'paragraphs':
        $info['entity_type'] = 'paragraphs_item';
        $field_settings = field_info_instance($entity_type, $field['field_name'], $bundle);
        $info['bundles'] = implode(', ', $field_settings["settings"]["allowed_bundles"]);
        break;

      // This field type references users
      case 'user_reference':
        $info['entity_type'] = 'user';
        $info['bundles'] = 'user';
        break;

      // This field type references nodes
      case 'node_reference':
        $info['entity_type'] = 'node';
        $node_types = $field["settings"]["referenceable_types"];
        $referenced_bundles = [];
        foreach ($node_types as $node_type) {
          if ($node_type != 0) {
            $referenced_bundles[] = $node_type;
          }
        }
        $info['bundles'] = implode(', ', $referenced_bundles);
        break;

      // This field type references commerce products
      case 'commerce_product_reference':
        $info['entity_type'] = 'commerce_product';
        $info['bundles'] = '';
        $field_settings = field_info_instance($entity_type, $field['field_name'], $bundle);
        if (isset($field_settings["settings"]["referenceable_types"]) && !empty($field_settings["settings"]["referenceable_types"])) {
          $referenced_bundles = [];
          foreach ($field_settings["settings"]["referenceable_types"] as $product_type => $product_type_selected) {
            if ($product_type_selected != 0) {

            }
            $referenced_bundles[] = $product_type;
          }
          $info['bundles'] = implode(', ', $referenced_bundles);
        } else {
          $info['bundles'] = implode(', ', array_keys(commerce_product_types()));
        }
        break;

      // This field type references commerce line item references
      case 'commerce_line_item_reference':
        $info['entity_type'] = 'commerce_line_item';
        $info['bundles'] = implode(', ', array_keys(commerce_line_item_types()));
        break;

      // This field type references commerce customer profile references
      case 'commerce_customer_profile_reference':
        $info['entity_type'] = 'commerce_customer_profile';
        $info['bundles'] = $field["settings"]["profile_type"];
        break;

      // This field type references commerce option set references
      case 'commerce_option_set_reference':
        $info['entity_type'] = 'commerce_option_set';
        $info['bundles'] = implode(', ', array_keys(commerce_option_get_sets()));
        break;

      // This field type references Views.
      case 'viewfield':
        $info['entity_type'] = 'viewfield';
        $field_settings = field_info_instance($entity_type, $field['field_name'], $bundle);
        $all_views = TRUE;
        $allowed_views = [];
        foreach ($field_settings["settings"]["allowed_views"] as $allowed_view => $allowed_view_status) {
          if ($allowed_view_status != 0) {
            $all_views = FALSE;
            $allowed_views[] = $allowed_view;
          }
        }
        if ($all_views) {
          $info['bundles'] = 'All views';
        } else {
          $info['bundles'] = implode(', ', $allowed_views);
        }
        break;
    }

    return $info;
  }

}
