<?php

/**
 * Class to export data about enabled modules.
 */
class AuditExportModulesEnabled extends AuditExportAuditData {

  /**
   * Stores update information for all projects.
   *
   * @var array
   */
  protected $updateInfo;

  /**
   * Indicates whether the Update Manager module is enabled.
   *
   * @var bool
   */
  protected $updateManagerEnabled;

  /**
   * Stores the current Drupal version.
   *
   * @var string
   */
  protected $drupalVersion;

  /**
   * Constructs the AuditExportModulesEnabled object.
   */
  public function __construct() {
    $this->setHeaders([
      'Module Name (machine_name)',
      'Current Version',
      'Latest Release',
      'Status',
      'Type',
      'Parent Module'
    ]);

    // Check for Update Manager module availability.
    $this->updateManagerEnabled = module_exists('update') && function_exists('update_get_available');
    $this->drupalVersion = VERSION;

    // Fetch update information if the Update Manager is enabled.
    if ($this->updateManagerEnabled) {
      $this->updateInfo = update_get_available();
    }
  }

  /**
   * Prepares data for export, focusing on enabled modules.
   *
   * @return array
   *   An array of machine names for enabled modules.
   */
  public function prepareData(): array {
    $modules = system_rebuild_module_data();
    $enabled_modules = array_filter($modules, function($module) {
      return !empty($module->status);
    });
    return array_keys($enabled_modules);
  }

  /**
   * Processes data for each module, enriching it with additional details.
   *
   * @param array $params
   *   Parameters, including the machine name of the module.
   *
   * @return array
   *   An array containing processed data for a module.
   */
  public function processData(array $params = []): array {
    $machine_name = $params['row_data'];
    $module_data = system_rebuild_module_data();
    if (isset($module_data[$machine_name])) {
      $module = $module_data[$machine_name];
      list($module_type, $parent_module) = $this->determineModuleType($module->name);
      $status = !empty($module->status) ? t('Enabled') : t('Disabled');
      $latest_release = $module_type === 'Core' ? $this->drupalVersion : $this->getLatestRelease($machine_name);
      return [
        $module->info['name'] . ' (' . $machine_name . ')',
        $module->info['version'],
        $latest_release,
        $status,
        $module_type,
        $parent_module,
      ];
    }

    return [];
  }

  /**
   * Determines the module type based on its .info file metadata.
   *
   * @param string $module_name
   *   The machine name of the module.
   *
   * @return array
   *   An array containing the type of the module and the associated parent module if it is a submodule.
   */
  private function determineModuleType($module_name) {
    $module_path = drupal_get_path('module', $module_name);
    $info_file_path = $module_path . "/$module_name.info";
    $parent_module = '';

    if (file_exists($info_file_path)) {
      $info_contents = file_get_contents($info_file_path);

      // Check for Core module marker.
      if (strpos($info_contents, '; Information added by Drupal.org packaging script') !== false) {
        if (preg_match('/^project\s*=\s*"(.+)"$/m', $info_contents, $matches)) {
          $project_name = $matches[1];
          if ($project_name === 'drupal') {
            return ['Core', $parent_module];
          }
          // If the module name matches the project name, it's a contributed module.
          if ($module_name === $project_name) {
            return ['Contributed', $parent_module];
          }
          // Otherwise, it's a submodule.
          return ['Submodule', $project_name];
        }
      }

      // Check if the parent directory is a module.
      $parent_directory = dirname($module_path);
      $parent_info_file = $parent_directory . '/' . basename($parent_directory) . '.info';

      if (file_exists($parent_info_file)) {
        $parent_module = basename($parent_directory);
        return ['Submodule', $parent_module];
      }

      // Check if the grandparent directory is a module.
      $grandparent_directory = dirname($parent_directory);
      $grandparent_info_file = $grandparent_directory . '/' . basename($grandparent_directory) . '.info';

      if (file_exists($grandparent_info_file)) {
        $parent_module = basename($grandparent_directory);
        return ['Submodule', $parent_module];
      }
    }

    return ['Custom', $parent_module];
  }

  /**
   * Fetches the latest release version for a module.
   *
   * @param string $machine_name
   *   The machine name of the module.
   *
   * @return string
   *   The latest release version or a status message.
   */
  private function getLatestRelease($machine_name) {
    // Direct comparison with 'drupal' is simplified for demonstration.
    // Adjust based on actual criteria for identifying core modules.
    if ($this->updateManagerEnabled && isset($this->updateInfo['projects'][$machine_name])) {
      // Placeholder for actual logic to determine the latest release.
      return '';
    }
    return $this->updateManagerEnabled ? t('N/A') : t('Update Manager Disabled');
  }
}
