<?php

/**
 * EntityFieldAudit class to gather field information for all entity types and bundles.
 */
class AuditExportSiteReport extends AuditExportAuditData {

  public function __construct() {
    $this->setHeaders([
      'Report',
      'Description',
      'Status',
    ]);
  }

  public function prepareData(): array {
    $reports = [];

    $reports[] = [
      'name' => 'php_version',
      'label' => 'PHP Version',
      'description' => 'Return PHP version that\'s currently installed on the environment.',
      'process' => 'checkPhpVersion'
    ];

    $reports[] = [
      'name' => 'ip_address',
      'label' => 'IP Address',
      'description' => 'Return the IP address of the environment.',
      'process' => 'checkIpAddress'
    ];

    $reports[] = [
      'name' => 'default_theme',
      'label' => 'Default theme',
      'description' => 'The current default theme for the site.',
      'process' => 'checkDefaultTheme'
    ];

    $reports[] = [
      'name' => 'admin_theme',
      'label' => 'Admin theme',
      'description' => 'The current admin theme for the site.',
      'process' => 'checkAdminTheme'
    ];

    $reports[] = [
      'name' => 'ecommerce',
      'label' => 'E-commerce',
      'description' => 'Check installed modules for known ecommerce configuration.',
      'process' => 'checkEcommerce'
    ];

    $reports[] = [
      'name' => 'git_version_control',
      'label' => 'Git version control',
      'description' => 'Check if git version control is detected.',
      'process' => 'checkGitHandle'
    ];

    $reports[] = [
      'name' => 'drupal_version',
      'label' => 'Drupal version',
      'description' => 'The current Drupal version for the site.',
      'process' => 'checkDrupalVersion'
    ];

    return $reports;
  }

  public function processData(array $params = []): array {
    $report_data = [];
    if (!empty($params["row_data"])) {
      if (!empty($params["row_data"]["process"])) {
        if (method_exists($this, $params["row_data"]["process"])) {
          $report_data = call_user_func([$this, $params["row_data"]["process"]]);
        }
      }
    }

    return [
      $params["row_data"]["label"],
      $params["row_data"]["description"],
      (!empty($report_data['status'])) ? $report_data['status'] : NULL,
    ];
  }

  /**
   * Return PHP version for current environment.
   *
   * @return array
   */
  private function checkPhpVersion(): array {

    $core_mods = [];

    foreach (ini_get_all('core') as $mod => $core_mod) {
      $core_mods[$mod] = $core_mod['local_value'];
    }

    return [
      'status' => PHP_VERSION,
    ];
  }

  /**
   * Return the IP address of the environment running the report.
   *
   * Tries to find the IP address from environment variables commonly set in cloud
   * hosting or containerized environments.
   *
   * @return array
   */
  private function checkIpAddress(): array {
    $possibleEnvVars = [
      'SERVER_ADDR',       // Commonly used by web servers
      'HOST_IP',           // Sometimes set in containerized environments
      'DOCKER_HOST_IP',    // Example for Docker environments, if explicitly set
    ];

    $ipAddress = 'Unknown'; // Default fallback

    foreach ($possibleEnvVars as $envVar) {
      if (!empty($_SERVER[$envVar])) {
        $ipAddress = $_SERVER[$envVar];
        break; // Exit the loop once we find a non-empty value
      }
    }

    return [
      'status' => $ipAddress,
    ];
  }

  /**
   * Return default theme for the site.
   *
   * @return array
   */
  private function checkDefaultTheme(): array {
    return [
      'status' => ucfirst(variable_get('theme_default', 'undefined')),
    ];
  }

  /**
   * Return admin theme for the site.
   *
   * @return array
   */
  private function checkAdminTheme(): array {
    return [
      'status' => ucfirst(variable_get('admin_theme', 'undefined')),
    ];
  }

  /**
   * Check if ecommerce modules are enabled.
   *
   * @return array
   */
  private function checkEcommerce(): array {
    $ecommerce_modules_enabled = FALSE;
    $details = [];
    $ecommerce_modules = [
      'commerce',
      'ubercart',
      'payment',
      'pay',
      'stripe_pay',
      'donate',
    ];

    foreach ($ecommerce_modules as $ecommerce_module) {
      if (module_exists($ecommerce_module)) {
        $ecommerce_modules_enabled = TRUE;
      }
      $details[$ecommerce_module] = [
        'enabled' => module_exists($ecommerce_module),
      ];
    }

    return [
      'status' => ($ecommerce_modules_enabled) ? 'Enabled' : 'Not detected',
    ];
  }


  /**
   * Detect if git version control is being used.
   *
   * @return string[]
   */
  function checkGitHandle(): array {
    // Set the maximum number of parent levels to search for the .git directory.
    $max_parent = 4;
    $git_detected = FALSE;
    $currentDir = DRUPAL_ROOT;

    // Iterate up to $max_parent levels to locate the .git directory.
    for ($i = 0; $i < $max_parent; $i++) {
      $gitPath = $currentDir . '/.git';
      if (file_exists($gitPath)) {
        $git_detected = TRUE;
        break;
      }
      $currentDir = dirname($currentDir);
    }

    // Return the search outcome.
    return [
      'status' => $git_detected ? 'Git detected' : 'Git not detected',
    ];

  }

  function checkDrupalVersion(): array {
    return [
      'status' => (defined('VERSION') ? VERSION : NULL),
    ];
  }

}
