<?php

/**
 * Gather information about Drupal views and their displays, including VBO usage and module source.
 */
class AuditExportViews extends AuditExportAuditData {

  public function __construct() {
    $this->setHeaders([
      'View Name',
      'Display Name',
      'Status',
      'Type',
      'Display Path',
      'Module Name',
      'VBO Usage'
    ]);
  }

  public function prepareData(): array {
    $views_and_displays = [];
    if (module_exists('views')) {
      $views_data = views_get_all_views();

      foreach ($views_data as $view) {
        $displays = array_keys($view->display);
        // Filter out Master if there are other displays
        if (count($displays) > 1 && isset($view->display['default'])) {
          unset($displays[array_search('default', $displays)]);
        }

        foreach ($displays as $display_id) {
          $views_and_displays[] = [
            'view_name' => $view->name,
            'display_id' => $display_id
          ];
        }
      }
    }

    return $views_and_displays;
  }

  public function processData(array $params = []): array {
    $view_name = $params['row_data']['view_name'];
    $display_id = $params['row_data']['display_id'];
    $view = views_get_view($view_name);
    $view->set_display($display_id);
    $display = $view->display[$display_id];
    $display_path = $display->display_options['path'] ?? '';

    $status = $view->disabled ? 'Disabled' : 'Enabled';
    $display_name = $display->display_title;
    $vbo_usage = $this->displayUsesVBO($view, $display_id);
    $module_name = $this->getModuleName($view_name);
    $type = $this->getViewType($view);

    return [
      $view->human_name . " ($view_name)",
      $display_name,
      $status,
      $type,
      $display_path,
      $module_name,
      (!empty($vbo_usage)) ? implode(', ', $vbo_usage) : NULL
    ];
  }
  /**
   * Check if the specific display uses Views Bulk Operations and list actions with their labels.
   *
   * @param object $view The view object loaded with the specific display.
   * @param string $display_id The display ID.
   * @return array Actions with their labels or module-defined labels if VBO is used, otherwise empty.
   */
  private function displayUsesVBO($view, $display_id): array {
    $actions = [];
    $all_actions = actions_get_all_actions(); // Ensure this function is available and returns the correct data structure

    foreach ($view->get_items('field', $display_id) as $item) {
      if (isset($item['field']) && $item['field'] === 'views_bulk_operations') {
        foreach ($item['vbo_operations'] as $action_id => $action_details) {
          // Checking for rules_component or regular actions
          if (strpos($action_id, 'rules_component::') === 0) {
            $machine_name = str_replace('rules_component::', '', $action_id);
            $suffix = '(Rules Component)';
          } else {
            $machine_name = str_replace('action::', '', $action_id);
            $suffix = '(Action)';
          }

          // Fetching the label either from VBO details or from global actions, prefixing appropriately
          $action_label = !empty($action_details['label']) ?
            "{$action_details['label']} {$suffix}" :
            (isset($all_actions[$machine_name]['label']) ?
              t($all_actions[$machine_name]['label']) . " {$suffix}":
              "{$machine_name} {$suffix}");

          $actions[] = $action_label;
        }
        return $actions;
      }
    }
    return $actions; // Return empty array if no VBO field is found
  }



  /**
   * Get the module name based on the view's machine name by searching through module files.
   *
   * @param string $view_name The view's machine name.
   * @return string The name of the module or an empty string if not found.
   */
  private function getModuleName($view_name): string {
    $modules = views_get_module_apis();
    foreach (array_keys($modules) as $module) {
      $path = drupal_get_path('module', $module);
      if ($this->searchModuleFiles($path, $view_name)) {
        return $module;
      }
    }
    return '';
  }

  /**
   * Recursively search module files for view definitions.
   *
   * @param string $module_path The path to the module directory.
   * @param string $view_name The name of the view to search for.
   * @return bool True if the view definition is found, false otherwise.
   */
  private function searchModuleFiles($module_path, $view_name) {
    $files = file_scan_directory($module_path, '/\.inc$/', ['recurse' => TRUE]);

    foreach ($files as $file) {
      $contents = file_get_contents($file->uri);
      // Check for both standard view export files and custom view definitions
      if (preg_match("/\\\$view\s*=\s*new\s+view\(\);/s", $contents) && preg_match("/\\\$view->name\s*=\s*'{$view_name}';/", $contents)) {
        return true;
      }
    }

    return false;
  }



  /**
   * Get the human-readable type of the view.
   *
   * @param object $view The view object.
   * @return string The type description.
   */
  private function getViewType($view): string {
    switch ($view->type) {
      case 'Normal':
        return t('Database');
      case 'Overridden':
        return t('Overridden');
      default:
        return t('In Code');
    }
  }
}
