<?php

class AuditExportAudit {

  /**
   * Machine name of the audit
   *
   * @var string
   */
  public string $name;

  /**
   * Human-readable name
   *
   * @var string
   */
  public string $label;

  /**
   * The name of the module providing the audit
   *
   * @var string
   */
  public string $module;
  /**
   * The name of the module providing the audit
   *
   * @var string
   */
  public string $data_type;

  /**
   * A field on the audit to serve as a unique identify / primary key.
   *
   * @var string
   */
  public string $identifier;

  /**
   * Audit group
   *
   * @var string
   */
  public string $group;

  /**
   * Audit dependencies
   *
   * @var array
   */
  public array $dependencies;

  /**
   * Audit configuration
   *
   * @var array
   */
  public array $configuration;

  /**
   * Description of the audit
   *
   * @var string
   */
  public string $description;

  /**
   * The className function for the audit.
   *
   * @var string
   */
  public string $className;

  /**
   * Retrieves audits based on the provided names.
   * If no names are provided, it returns all audits.
   *
   * @param array $names
   *   The names of the audits to retrieve. If empty, all audits are retrieved.
   *
   * @return array
   *   An array of AuditExportAudit instances.
   * @throws \Exception
   */
  public static function getAudits(array $names = []): array {
    if (!self::validate()) {
      throw new Exception('Audit name validation failed. Duplicate names found.');
    }

    $audit_data = module_invoke_all('aex_audit');
    $requested_audits = [];

    // If $names is empty, return all audits.
    if (empty($names)) {
      foreach ($audit_data as $data) {
        $skip_audit = FALSE;
        if ($data["dependencies"]) {
          foreach ($data["dependencies"] as $dependency) {
            if (!module_exists($dependency)) {
              $skip_audit = TRUE;
            }
          }
        }

        if (!$skip_audit) {
          if (empty($data['data_type'])) {
            $data['data_type'] = 'flat';
          }
          $audit = self::buildAuditData($data);
          if ($audit) {
            $requested_audits[$data['name']] = $audit;
          }
        }
      }
      return $requested_audits;
    }

    // Otherwise, return only the audits specified in $names.
    foreach ($audit_data as $data) {
      if (in_array($data['name'], $names, true)) {
        $audit = self::buildAuditData($data);
        if ($audit) {
          $requested_audits[$data['name']] = $audit;
        }
      }
    }

    return $requested_audits;
  }

  /**
   * Checks if all dependencies are enabled.
   *
   * @return bool
   *   True if all dependencies are enabled, false otherwise.
   */
  public function checkDependencies(): bool {
    foreach ($this->dependencies as $module) {
      if (!module_exists($module)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Helper function to create an audit instance from data.
   *
   * @param array $data
   *   Data array for the audit.
   *
   * @return AuditExportAudit|null
   *   An instance of AuditExportAudit or null if data is not valid.
   */
  protected static function buildAuditData(array $data): ?AuditExportAudit {
    if (
      isset($data['name'], $data['label'], $data['module'], $data['data_type'], $data['identifier'],
        $data['group'], $data['dependencies'], $data['description'],
        $data['className'])
    ) {
      $audit = new self();
      $audit->name = $data['name'];
      $audit->label = $data['label'];
      $audit->module = $data['module'];
      $audit->data_type = $data['data_type'];
      $audit->identifier = $data['identifier'];
      $audit->group = $data['group'];
      $audit->dependencies = $data['dependencies'];
      $audit->description = $data['description'];
      $audit->className = $data['className'];
      return $audit;
    }
    return null;
  }

  /**
   * Validates audit instances to ensure unique names.
   *
   * @return bool
   *   Returns true if all audits have unique names, false otherwise.
   */
  public static function validate(): bool {
    $audit_data = module_invoke_all('aex_audit');
    $names = array();

    // Check fore duplicate audits.
    foreach ($audit_data as $data) {
      if (isset($data['name'])) {
        if (in_array($data['name'], $names)) {
          return false;
        }
        $names[] = $data['name'];
      }
    }

    return true;
  }

  /**
   * Return last time the
   *
   * @param bool $raw
   *
   * @return string
   */
  public function getLastProcessDate(bool $raw = FALSE): string {
    $auditDataInstance = new $this->className();
    $auditDataInstanceData = $auditDataInstance->getData($this);
    if (!empty($auditDataInstanceData["date"])) {
      if ($raw) {
        $text = $auditDataInstanceData["date"];
      } else {
        $text = format_date($auditDataInstanceData["date"], 'short');
      }
    } else {
      $text = "Not processed";
    }
    return $text;
  }


}
