<?php

class AuditExportAuditData {

  // Define class constants for fixed values of $data_type
  const DATA_TYPE_FLAT = 'flat';
  const DATA_TYPE_CROSS = 'cross';
  const DATA_TYPE_PROCESS = 'process';

  /**
   * @var array
   */
  protected array $data_headers = [];


  /**
   * @var string
   */
  protected string $crossTab_header_label = '';

  /**
   * @var string
   */
  protected string $data_type;

  /**
   * Constructor that initializes the data type.
   *
   * @param string|null $type Optional. The data type to be used. If
   * not provided, DATA_TYPE_FLAT is used as a default.
   */
  public function __construct(string $type = null) {
    $this->setDataType($type ?? self::DATA_TYPE_FLAT);
  }

  /**
   * @param array $headers
   *
   * @return void
   */
  public function setHeaders(array $headers) {
    $this->data_headers = $headers;
  }

  /**
   * @param string $crossTab_header_label
   *
   * @return void
   */
  public function setCrossTabHeaderLabel(string $crossTab_header_label) {
    $this->crossTab_header_label = $crossTab_header_label;
  }

  /**
   *
   * @return string
   */
  public function getCrossTabHeaderLabel(): string {
    return $this->crossTab_header_label;
  }

  /**
   * Return audit override configuration for the report.
   *
   * @param $audit
   *
   * @return array
   */
  public function getConfig($audit): array {
    $config = [];

    $option_overrides = [
      'audit_export_override',
      'audit_export_enable_cron',
      'audit_export_cron_queue_timeout',
      'audit_export_cron_frequency_default',
    ];
    foreach ($option_overrides as $override_option) {
      $config[$override_option . '_' . $audit->name] =
        (string) variable_get($override_option . '_' . $audit->name);
    }

    return $config;
  }

  /**
   * @return array
   */
  public function getHeaders(): array {
    // Add CrossTabHeaderLabel, if available.
    if (!empty($this->getCrossTabHeaderLabel())) {
      $headers[] = $this->getCrossTabHeaderLabel();
      foreach ($this->data_headers as $header) {
        $headers[] = $header;
      }
      return $headers;
    }

    return $this->data_headers;
  }

  /**
   * Sets the data type if it matches one of the predefined constants.
   *
   * @param string $type
   *
   * @throws InvalidArgumentException If the type is not one of the predefined constants.
   */
  public function setDataType(string $type) {
    $allowedTypes = [self::DATA_TYPE_CROSS, self::DATA_TYPE_FLAT, self::DATA_TYPE_PROCESS];
    if (!in_array($type, $allowedTypes)) {
      throw new InvalidArgumentException("Invalid data type. Allowed types are: " . implode(", ", $allowedTypes));
    }
    $this->data_type = $type;
  }

  /**
   * @return string
   */
  public function getDataType(): string {
    return $this->data_type;
  }

  /**
   * @param $audit
   *
   * @return array
   */
  public function getData($audit): array {
    return (new AuditExportAuditReport($audit, $this))->getLastReport($audit);
  }

  /**
   * @return array
   */
  public function prepareCrossTabScripts(): array {
    return [];
  }

  /**
   * @return array
   */
  public function prepareData(): array {
    return [];
  }

  /**
   * @param array $params
   *
   * @return array
   */
  public function processData(array $params = []): array {
    return [];
  }

  /**
   * @return array
   */
  public function processDataPreProcess(): array {
    return [];
  }

}
