<?php

/**
 * Page callback for the Overview report.
 *
 * @throws \Exception
 */
function audit_export_core_reports_overview() {
  $header = array(t('Audit'),t('Group'),t('Description'),t('Last processed'),t('Actions'));
  $audits = AuditExportAudit::getAudits();
  $options = ['empty' => t('No audits available.')];
  $rows = [];

  // Prepare rows for the table.
  foreach ($audits as $audit) {
    $audit_link_base = 'admin/reports/audit-export/reports/';
    $audit_link = l(t($audit->label), $audit_link_base . $audit->group . '/' . $audit->name);
    $process_link = l(t('Run audit'), $audit_link_base . $audit->group . '/' . $audit->name . '/process');
    $group_link = l(t($audit->group), $audit_link_base . $audit->group);

    $rows[] = array(
      $audit_link,
      $group_link,
      check_plain($audit->description),
      $audit->getLastProcessDate(),
      $process_link,
    );
  }

  // Utilize AuditExportDisplay for rendering the table.
  return AuditExportDisplay::renderDisplay($header, $rows, $options);
}

/**
 * @param string $audit
 *
 * @return string
 * @throws \Exception
 */
function audit_export_core_audit_view(string $audit): string {
  $audit = AuditExportAudit::getAudits()[$audit];
  $options['empty'] = t("The $audit->name audit has not yet been processed or has no data.");
  $output = '';
  $headers = [];
  $rows = [];
  $formatted_date = '';
  $report_link = '';

  drupal_set_title(check_plain($audit->label));
  module_load_include($audit->className . '.inc', $audit->module, 'inc');

  if (class_exists($audit->className)) {
    $auditDataInstance = new $audit->className();
    $auditDataInstanceData = $auditDataInstance->getData($audit);
    if (method_exists($auditDataInstance, 'getHeaders')) {
      $config = $auditDataInstance->getConfig($audit);
      foreach ($auditDataInstance->getHeaders() as $header) {
        $headers[] = $header;
      }
    }
    if (!empty($auditDataInstanceData['data'])) {
      $rows = unserialize($auditDataInstanceData['data']);
      $rows = AuditExportDisplay::tokenDisplayCleanup($rows);
    }

    if (!empty($auditDataInstanceData["fid"])) {
      $file = file_load($auditDataInstanceData["fid"]);
      $url = file_create_url($file->uri);
      $report_link = l(t('Download CSV'), $url, array('attributes' => array('target' => '_blank')));
    }

    if (!empty($auditDataInstanceData["date"])) {
      $formatted_date = format_date($auditDataInstanceData["date"], 'short');
    }

    // Prepare the header with the group description.
    $output .= sprintf("<p>%s</p>", check_plain($audit->description));
    (!empty($formatted_date && $report_link) ?
      $output.= sprintf(
        "<div>%s</div>",
        t("Last updated: $formatted_date | $report_link")) :
      NULL
    );
    $output .= AuditExportDisplay::renderDisplay($headers, $rows, $options);
  }

  return $output;
}

/**
 * @param $form
 * @param $form_state
 * @param $audit_name
 *
 * @return array
 * @throws \Exception
 */
function audit_export_core_audit_config_form($form, &$form_state, $audit_name) {
  $audit = AuditExportAudit::getAudits()[$audit_name];
  $form['#attached']['audit'] = $audit;

  // Override audit settings
  $form['custom_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Override global settings'),
    '#collapsible' => FALSE,
    '#collapsed' => !(bool) variable_get('audit_export_override_' . $audit_name, 0),
  );

  $form['custom_settings']['override_global_settings'] = array(
    '#type' => 'checkbox',
    '#title' => t('Override global settings for this audit'),
    '#default_value' => variable_get('audit_export_override_' . $audit_name, 0),
    '#description' => t('Enable this option to customize the settings for this individual audit.'),
  );

  $form['custom_settings']['custom_enable_report_cron'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable report processing on cron'),
    '#default_value' => variable_get('audit_export_enable_cron_' . $audit_name, variable_get('audit_export_enable_cron', 0)),
    '#description' => t('Enable this option to allow reports to be processed on cron.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="override_global_settings"]' => array('checked' => TRUE),
      ),
    ),
  );

  $form['custom_settings']['custom_default_frequency'] = array(
    '#type' => 'textfield',
    '#title' => t('Default frequency'),
    '#default_value' => variable_get('audit_export_cron_frequency_default_' . $audit_name, variable_get('audit_export_cron_frequency_default', 1440)),
    '#size'=> 15,
    '#description' => t('The number of minutes that must pass before a report is refreshed.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="override_global_settings"]' => array('checked' => TRUE),
      ),
    ),
  );

  $form['custom_settings']['custom_queue_timeout'] = array(
    '#type' => 'textfield',
    '#title' => t('Queue timeout'),
    '#default_value' => variable_get('audit_export_cron_queue_timeout_' . $audit_name, variable_get('audit_export_cron_queue_timeout', 120)),
    '#size'=> 15,
    '#description' => t('The timeout for the queue when reports are refreshed on cron.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="override_global_settings"]' => array('checked' => TRUE),
      ),
    ),
  );

  // Custom theme for table layout.
  $form['#theme'] = 'audit_export_configuration_table';

  $form['submit'] = [
    '#type' => 'submit',
    '#value' => t('Save Configuration'),
  ];

  return $form;
}

/**
 * @param $form
 * @param $form_state
 *
 * @return void
 */
function audit_export_core_audit_config_form_submit($form, &$form_state) {
  $audit_name = $form["#attached"]["audit"]->name;

  // Display configuration
  $override = $form_state['values']['override_global_settings'];
  variable_set('audit_export_override_' . $audit_name, $override);
  if ($override) {
    // Save custom settings
    variable_set('audit_export_enable_cron_' . $audit_name, $form_state['values']['custom_enable_report_cron']);
    variable_set('audit_export_cron_frequency_default_' . $audit_name, $form_state['values']['custom_default_frequency']);
    variable_set('audit_export_cron_queue_timeout_' . $audit_name, $form_state['values']['custom_queue_timeout']);

  } else {
    // Clear custom settings to ensure they don't affect global fallback
    variable_del('audit_export_enable_cron_' . $audit_name);
    variable_del('audit_export_cron_frequency_default_' . $audit_name);
    variable_del('audit_export_cron_queue_timeout_' . $audit_name);
  }
}


/**
 * Page callback for individual groups of Audit Export Reports.
 *
 * @param string $group_name
 *   The group names.
 *
 * @return string
 *   The HTML output for the group page.
 * @throws \Exception
 */
function audit_export_core_group_view(string $group_name): string {
  $audits = AuditExportAudit::getAudits();
  $groups = AuditExportAuditGroup::getAuditGroups();
  $header = array(t('Audit'), t('Description'), t('Last processed'), t('Actions'));
  $options['empty'] = t('No audits available for this group.');
  $rows = [];
  $output = '';

  // Prepare the header with the group description
  foreach ($groups as $group) {
    if ($group->name == $group_name) {
      drupal_set_title(t(check_plain($group->label)));
      $output .= sprintf("<p>%s</p>", check_plain($group->description));

      // Filter audits by group
      foreach ($audits as $audit) {
        if ($audit->group == $group_name) {
          $audit_link_base = 'admin/reports/audit-export/reports/';
          $process_link = l(t('Run audit'), $audit_link_base . $audit->group . '/' . $audit->name . '/process');
          $audit_link = l(
            t($audit->label),
            $audit_link_base . $audit->group . '/' . $audit->name
          );
          $rows[] = array(
            $audit_link,
            t($audit->description),
            $audit->getLastProcessDate(),
            $process_link
          );
        }
      }
      break;
    }
  }

  $output .= AuditExportDisplay::renderDisplay(
    $header,
    $rows,
    $options
  );

  return $output;
}

/**
 * Page callback for the settings page.
 */
function audit_export_core_settings_page($form, &$form_state) {
  // Check if the private filesystem is configured.
  $private_filesystem_configured = variable_get('file_private_path', '') ? TRUE : FALSE;

  // Start with only the Public filesystem option.
  $options = array(
    'temporary' => t('Temporary'),
    'public' => t('Public'),
  );

  // Add the Private filesystem option only if it's configured.
  if ($private_filesystem_configured) {
    $options['private'] = t('Private');
  }

  $form['report_file_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('CSV Download'),
    '#collapsible' => TRUE,
    '#collapsed' => !(bool) variable_get('audit_export_save_filesystem', 0),
  );

  $form['report_file_settings']['save_to_filesystem'] = array(
    '#type' => 'checkbox',
    '#title' => t('Save export to filesystem'),
    '#default_value' => variable_get('audit_export_save_filesystem', 0),
    '#description' => t('Enable this option to save exports to the filesystem.'),
  );

  $form['report_file_settings']['audit_export_filesystem'] = array(
    '#type' => 'radios',
    '#title' => t('Filesystem'),
    '#options' => $options,
    '#default_value' => variable_get('audit_export_filesystem', 'temporary'),
    '#description' => t('Select where the export files should be stored. Using the private filesystem is recommended if available.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="save_to_filesystem"]' => array('checked' => TRUE),
      ),
    ),
    '#ajax' => array(
      'callback' => 'audit_export_core_filesystem_ajax_callback',
      'wrapper' => 'audit-export-path-wrapper',
    ),
  );

  // Define the wrapper and field once correctly.
  $form['report_file_settings']['audit_export_path_wrapper'] = array(
    '#type' => 'container',
    '#attributes' => array('id' => 'audit-export-path-wrapper'),
  );

  // Initialize field prefix based on the default or previously saved value.
  $default_prefix = variable_get('audit_export_filesystem', 'public') . '://';

  $form['report_file_settings']['audit_export_path_wrapper']['audit_export_path'] = array(
    '#type' => 'textfield',
    '#title' => t('Export path'),
    '#size'=> 40,
    '#field_prefix' => t($default_prefix),
    '#default_value' => variable_get('audit_export_filesystem_path', 'audit-export'),
    '#description' => t('Enter the path where audit exports will be stored on the filesystem.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="save_to_filesystem"]' => array('checked' => TRUE),
      ),
    ),
  );

  // Group for Report Configuration.
  $form['cron_configuration'] = array(
    '#type' => 'fieldset',
    '#title' => t('Cron Configuration'),
    '#collapsible' => TRUE,
    '#collapsed' => !(bool) variable_get('audit_export_enable_cron', 0),
  );

  $form['cron_configuration']['enable_report_cron'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable report processing on cron'),
    '#default_value' => variable_get('audit_export_enable_cron', 0),
    '#description' => t('Enable this option to allow reports to be processed on cron.'),
  );

  $form['cron_configuration']['default_frequency'] = array(
    '#type' => 'textfield',
    '#title' => t('Default frequency'),
    '#default_value' => variable_get('audit_export_cron_frequency_default', 1440),
    '#size'=> 15,
    '#description' => t('The number of minutes that must pass before a report is refreshed.'),
    '#states' => array(
      'enabled' => array(
        ':input[name="enable_report_cron"]' => array('checked' => TRUE),
      ),
    ),
  );

  $form['cron_configuration']['queue_timeout'] = array(
    '#type' => 'textfield',
    '#title' => t('Queue timeout'),
    '#default_value' => variable_get('audit_export_cron_queue_timeout', 120),
    '#size'=> 15,
    '#description' => t('When reports are refreshed on cron, they use the Queue API. What value would you like to set for the timeout for the queue?'),
    '#states' => array(
      'enabled' => array(
        ':input[name="enable_report_cron"]' => array('checked' => TRUE),
      ),
    ),
  );

  $form['submit'] = array(
    '#type' => 'submit',
    '#weight' => 100,
    '#value' => t('Save settings'),
  );

  return $form;
}

/**
 * Form submission handler for audit_export_core_settings_page.
 */
function audit_export_core_settings_page_submit($form, &$form_state) {
  variable_set('audit_export_save_filesystem', $form_state["values"]["save_to_filesystem"]);
  variable_set('audit_export_filesystem', $form_state["values"]["audit_export_filesystem"]);
  variable_set('audit_export_filesystem_path', $form_state["values"]["audit_export_path"]);
  variable_set('audit_export_enable_cron', $form_state["values"]["enable_report_cron"]);
  variable_set('audit_export_cron_frequency_default', $form_state["values"]["default_frequency"]);
  variable_set('audit_export_cron_queue_timeout', $form_state["values"]["queue_timeout"]);
  drupal_set_message(t('The settings have been saved.'));
}

function audit_export_core_filesystem_ajax_callback($form, &$form_state) {
  // Determine the prefix based on the selected option.
  $selected_filesystem = $form_state['values']['audit_export_filesystem'];
  $prefix = $selected_filesystem . '://';

  // Update the field prefix dynamically.
  $form['report_file_settings']['audit_export_path_wrapper']['audit_export_path']['#field_prefix'] = t($prefix);

  // Return the updated wrapper for AJAX replacement.
  return $form['report_file_settings']['audit_export_path_wrapper'];
}
