<?php

/**
 * Build report on user roles and associated permissions.
 */
class AuditExportActiveUsersRoles extends AuditExportAuditData {

  /**
   * Constructor for AuditExportActiveUsersRoles class.
   */
  public function __construct() {
    $this->setHeaders($this->buildUserRoleHeaders());
    $this->setCrossTabHeaderLabel('Role Name');
  }

  /**
   * @return array
   */
  public function prepareCrossTabScripts(): array {
    $scripts['user_role_count'] = [
      'label' => '# of users',
      'name' => 'user_role_count',
      'method' => 'userRoleCount',
      'roles' => user_roles(),
    ];
    return $scripts;
  }

  /**
   * @return array
   */
  public function prepareData(): array {
    return $this->getPermissionModules();
  }

  /**
   * Build row data for report
   *
   * @param array $params
   *
   * @return array
   *
   */
  public function processData(array $params = []): array {
    $module = $params['row_data'];
    $output = [$this->permissionAudit(1, $module)['module_header']];
    foreach (array_keys(user_roles()) as $rid) {
      $role_data[] = $this->permissionAudit($rid, $module);
    }

    if (!empty($role_data)) {
      foreach ($role_data as $role_datum) {
        if (!empty($role_datum) && !empty($role_datum["permissions"])) {
          if ((count($module["permissions"]) == count($role_datum["permissions"]))) {
            $output[] = t('All permissions');
          } else {
            $output[] = $this->permissionListClean(implode(', ', $role_datum["permissions"]));
//            $output[] = implode(', ', $role_datum["permissions"]);
          }

        } else {
          $output[] = t('None');
        }
      }
    }

    return $output;
  }

  /**
   * Use preprocessDataPreProcess to add module permission header row.
   *
   * @return array
   */
  public function processDataPreProcess(): array {
    return [[
      'name' => 'user_role_perms_data_heading',
      'method' => 'modulePermsDataHeading',
    ]];
  }

  /**
   * Return an array of role id's to be used for the header values of the report.
   *
   * @return array
   */
  private function buildUserRoleHeaders(): array {
    // Fetch permissions for all roles or the one selected role.
    $user_roles = [];
    if (!empty(user_roles())) {
      foreach (user_roles() as $rid => $role) {
        $user_role = user_role_load($rid);
        $user_roles[] = $user_role->name;
      }
    }
    return $user_roles;
  }

  /**
   * Return a heading row for the module permission list.
   *
   * @return string[]
   */
  public function modulePermsDataHeading(): array {
    $data = ['Module', 'Permissions'];
    $row_cell_diff = (count(user_roles()) - count($data));
    for ($cell_count = 0; $cell_count <= $row_cell_diff;  $cell_count++) {
      $data[] = '';
    }
    return $data;
  }

  /**
   * Clean, aggregate, and retain all permission items, including those not matching dynamic slug patterns.
   *
   * @param string $permission_item A string containing various permissions.
   *
   * @return string A formatted string aggregating permissions by action, including non-matching items.
   */
  private function permissionListClean(string $permission_item): string {
    // Patterns to identify slugs and actions. Using placeholders for dynamic parts.
    $patterns = [
      '/Execute (.+)/',
      '/(.+): Create new content/',
      '/(.+): Edit own content/',
      '/(.+): Edit any content/',
      '/(.+): Delete own content/',
      '/(.+): Delete any content/',
      '/Create own value for field (.+)/',
      '/Edit anyone\'s value for field (.+)/',
      '/Edit own value for field (.+)/',
      '/View anyone\'s value for field (.+)/',
      '/View own value for field (.+)/',
      '/See environment indicator for (.+)/',
      '/Create (.+) products/',
      '/Edit any (.+) product/',
      '/View any (.+) product/',
      '/View own (.+) products/',
      '/Create (.+) customer profiles/',
      '/Edit any (.+) customer profile/',
      '/Edit own (.+) customer profile/',
      '/View any (.+) customer profile/',
      '/View own (.+) customer profile/',
    ];

    // Initialize an array to hold the aggregated permissions.
    $aggregatedPermissions = [];
    $unmatchedItems = [];

    // Explode the original permission string into an array.
    $permission_items = explode(', ', $permission_item);

    // Loop through each permission item.
    foreach ($permission_items as $item) {
      $matched = false; // Flag to track if the item matches any pattern.

      // Check against each pattern.
      foreach ($patterns as $pattern) {
        if (preg_match($pattern, $item, $matches)) {
          $matched = true; // Found a match.
          $action = str_replace($matches[1], '%slug%', $item);
          // Extract the slug (dynamic part) from the item.
          $slug = $matches[1];

          // Aggregate the slugs by the action.
          $aggregatedPermissions[$action][] = $slug;
          break; // Found a match, no need to check further patterns.
        }
      }

      // If no pattern matched, add the item to unmatchedItems.
      if (!$matched) {
        $unmatchedItems[] = $item;
      }
    }

    // Initialize a string to hold the final formatted output.
    $output = [];

    // Format the aggregated permissions into the desired output string.
    foreach ($aggregatedPermissions as $pattern => $slugs) {
      // Replace the placeholder with the actual action, removing the slug part.
      $action = str_replace('%slug%', '', $pattern);
      $output[] = trim($action, ': ') . ': ' . implode(', ', array_unique($slugs));
    }

    // Add unmatched items back to the output.
    $output = array_merge($output, $unmatchedItems);

    sort($output);

    // Return the formatted string, joining each action's aggregation with PHP_EOL.
    return implode('; ', $output);
  }

  /**
   * Generates an audit array of permissions by module for a given role or all roles.
   *
   * @param int $rid
   * The role ID to filter permissions by, or NULL for all roles.
   *
   * @param array|null $module
   *
   * @return array An associative array of permissions organized by module and roles.
   */
  private function permissionAudit(int $rid = 0, array $module = NULL): array {
    if (!is_numeric($rid) || !in_array($rid, array_keys(user_roles()))) {
      return array('error' => 'Invalid or non-existent role ID provided.');
    }

    $role_permissions = user_role_permissions(array($rid => user_roles(TRUE, $rid)));
    $modules_info = system_get_info('module');
    $permission_data = array();
    $audit = array();

    // Check if a specific module is requested.
    if ($module !== NULL && !module_exists($module["source_module"])) {
      return array('error' => 'Specified module does not exist or does not define permissions.');
    }

    $permissions = module_invoke($module["source_module"], 'permission');
    if ($permissions) {
      foreach ($permissions as $perm => $perm_item) {
        if (isset($role_permissions[$rid][$perm])) {
          // Only add if permission is relevant to the specified role.
          $permission_data[$module["source_module"]]['permissions'][$perm] =
            array(
              'title' => $perm_item['title'],
              'description' => $perm_item['description'] ?? '',
              'granted' => TRUE,
            );
        }
      }
      if (!empty($permission_data[$module["source_module"]])) {
        $permission_data[$module["source_module"]]['display_name'] =
          $modules_info[$module["source_module"]]['name'];
      }
    }

    $audit['module_header'] = $modules_info[$module["source_module"]]["name"];
    if (!empty($permission_data)) {
      $perms_output = [];
      foreach ($permission_data[$module["source_module"]]["permissions"] as $permission) {
        $perms_output[] = $permission["title"];
      }
      $audit['permissions'] = $perms_output;
    }

    return $audit;

  }

  /**
   * Retrieves all modules with their permissions.
   *
   * @return array An associative array of modules and their permissions.
   */
  private function getPermissionModules(): array {
    $modules_info = system_get_info('module');
    $permissions_array = array();

    // Retrieve all modules implementing permissions.
    foreach (module_implements('permission') as $module) {
      $permissions = module_invoke($module, 'permission');
      if ($permissions) {
        // Add permission title and description to the permission subarray.
        foreach ($permissions as $perm => $perm_item) {
          $permissions_array[$module]['source_module'] = $module;
          $permissions_array[$module]['permissions'][$perm] = array(
            'title' => $perm_item['title'],
            'description' => $perm_item['description'] ?? '',
          );
        }
      }
    }

    // Sort modules by their human-readable name.
    uasort($permissions_array, function ($a, $b) use ($modules_info) {
      $a_name = $modules_info[key($a)]['name'] ?? '';
      $b_name = $modules_info[key($b)]['name'] ?? '';
      return strcmp($a_name, $b_name);
    });

    return $permissions_array;
  }

  /**
   * Return the number of users assigned to a role.
   *
   * @param $roles
   * An array of user role id's.
   *
   * @return array
   * An array with counts for all users assigned to roles.
   */
  public function userRoleCount($roles): array {
    $row_data = [];

    foreach ($this->prepareCrossTabScripts() as $crossTabScript) {
      if (isset($crossTabScript['method']) &&
        $crossTabScript['method'] === __FUNCTION__) {
        if (isset($crossTabScript['label'])) {
          $row_data[] = $crossTabScript['label'];
        }
      }
    }

    foreach ($roles["roles"] as $rid => $role_name) {
      $query = db_select('users_roles', 'ur');
      $query->join('users', 'u', 'ur.uid = u.uid');
      $query->condition('ur.rid', $rid, '=')
        ->condition('u.status', 1, '=');

      // Convert to a count query
      $count = $query->countQuery()->execute()->fetchField();
      $row_data[] = number_format($count);
    }
    return $row_data;
  }

}
