<?php

/**
 * Gather information about content types.
 */
class AuditExportContentType extends AuditExportAuditData {

  public function __construct() {
    $this->setHeaders([
      'Content Type Name',
      'Machine Name',
      'Published Nodes',
      'Unpublished Nodes',
      'Enabled Displays'
    ]);
  }

  public function prepareData(): array {
    $content_types = node_type_get_types();
    $machine_names = [];
    foreach ($content_types as $machine_name => $type) {
      $machine_names[] = $machine_name;
    }
    return $machine_names;
  }

  public function processData(array $params = []): array {
    $machine_name = $params["row_data"];
    $type = node_type_get_type($machine_name);
    $published_count = $this->getNodeCountByStatus($machine_name, 1);
    $unpublished_count = $this->getNodeCountByStatus($machine_name, 0);
    $enabled_displays = $this->getEnabledDisplays($machine_name);

    return [
      $type->name,
      $machine_name,
      $published_count,
      $unpublished_count,
      is_array($enabled_displays) ? implode(', ', $enabled_displays) : $enabled_displays,
    ];
  }

  protected function getNodeCountByStatus($type, $status) {
    return db_select('node', 'n')
      ->fields('n', ['nid'])
      ->condition('type', $type)
      ->condition('status', $status)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Returns either an array or a string of enabled displays depending on the format_list parameter.
   *
   * @param string $type Content type machine name.
   * @param bool $format_list Whether to format the list as a string.
   *
   * @return mixed Can return either an array or a string.
   */
  protected function getEnabledDisplays($type, $format_list = FALSE) {
    $entity_info = entity_get_info('node');
    $enabled_displays = ['default' => t('Default')];
    $displays = isset($entity_info['bundles'][$type]['view modes']) ? array_keys($entity_info['bundles'][$type]['view modes']) : [];

    foreach ($displays as $display) {
      if (!empty($entity_info['bundles'][$type]['view modes'][$display]['custom settings'])) {
        $enabled_displays[] = $display;
      }
    }

    foreach ($entity_info["view modes"] as $machine => $view_mode) {
      if (isset($view_mode["custom settings"]) && $view_mode["custom settings"]) {
        $enabled_displays[$machine] = $view_mode["label"];
      }
    }

    if ($format_list) {
      return $this->formatList($enabled_displays);
    }

    return $enabled_displays;
  }

  /**
   * Format an array as an HTML list.
   *
   * @param array $array List of items to format.
   *
   * @return string Formatted list as HTML.
   */
  private function formatList($array): string {
    $listHtml = '<ul>';
    foreach ($array as $item) {
      if ($item !== '') {
        $listHtml .= "<li>" . check_plain($item) . "</li>";
      }
    }
    $listHtml .= '</ul>';

    return $listHtml;
  }
}
