<?php

/**
 * Class for auditing menu data and associated configurations within Drupal.
 */
class AuditExportMenus extends AuditExportAuditData {

  /**
   * Constructor for AuditExportMenus.
   * Initializes the headers for the audit export.
   */
  public function __construct() {
    $this->setHeaders([
      'Menu', 'Menu Block', 'Module', 'Links', 'Menu Usage',
      'Starting level', 'Max depth', 'Has expanded children',
    ]);
  }

  /**
   * Prepares the menu data for export by identifying each menu and its
   * corresponding module.
   *
   * @return array
   *   An array containing menu data.
   */
  public function prepareData(): array {
    // Retrieve all enabled and system-defined menus.
    $menus = array_keys(menu_get_menus());
    $system_menus = menu_list_system_menus();

    // Custom module menus, excluding system menus.
    $module_menus = $this->menusFromModules(
      array_diff($menus, array_keys($system_menus))
    );

    $return = [];
    foreach ($menus as $menu) {
      $matched_module = NULL;

      // Identify if the menu is a system menu.
      if (in_array($menu, array_keys($system_menus))) {
        $matched_module = 'system';
      } else {
        // Check against custom module menus.
        foreach ($module_menus as $module => $moduleMenuNames) {
          if (in_array($menu, $moduleMenuNames)) {
            $matched_module = $module;
            break; // Stop once a match is found.
          }
        }
      }

      $return[] = [
        'menu' => $menu,
        'module' => (!empty($matched_module)) ? $matched_module : 'undefined',
      ];
    }

    return $return;
  }

  /**
   * Processes menu data to include detailed information like links, menu
   * usage, etc.
   *
   * @param array $params
   *   Parameters, including 'row_data' with the menu name.
   *
   * @return array
   *   An array of processed menu data for export.
   */
  public function processData(array $params = []): array {
    // Load menu to fetch its title and other details.
    $menu_name = $params["row_data"]["menu"];
    $menu = menu_load($menu_name);

    // Retrieve menu links and blocks related to the menu.
    $menu_links = $this->getMenuLinks($menu_name);
    $blocks = $this->getMenuBlocks($menu_name);

    // Aggregate detailed information about the menu and blocks.
    $menu_info = $this->aggregateMenuInfo($menu_links);
    $block_info = $this->aggregateBlockInfo($blocks, $menu);

    return [
      $menu["title"] . " ($menu_name)",
      $block_info["title"],
      $params["row_data"]["module"],
      count($menu_links),
      (is_array($block_info['enabled_themes'])) ?
        implode(', ', $block_info['enabled_themes']) :
        $block_info['enabled_themes'],
      $menu_info['start_depth'] ?? t('N/A'),
      $menu_info['max_depth'] ?? t('N/A'),
      $menu_info['has_children'] ? t('Yes') : t('No'),
    ];
  }

  /**
   * Retrieves menu links for a specific menu.
   *
   * @param string $menu_name
   *   The menu name.
   *
   * @return array
   *   An array of menu link objects.
   */
  private function getMenuLinks(string $menu_name): array {
    return db_query
    ("SELECT menu_name, mlid, plid, options, module, has_children, expanded, depth
          FROM {menu_links}
          WHERE menu_name = :menu_name",
      array(':menu_name' => $menu_name)
    )->fetchAll();
  }

  /**
   * Retrieves blocks that reference a specific menu.
   *
   * @param string $menu_name
   *   The menu name.
   *
   * @return array
   *   An array of block objects.
   */
  private function getMenuBlocks(string $menu_name): array {
    return db_query
    ("SELECT bid, status, region, custom, visibility, pages, delta, module, theme, title, cache
        FROM {block}
        WHERE delta = :delta
        ",
      array(':delta' => $menu_name)
    )->fetchAll();
  }

  /**
   * Aggregates information from menu links to provide menu details.
   *
   * @param array $menu_links
   *   An array of menu link objects.
   *
   * @return array
   *   A structured array containing menu information.
   */
  private function aggregateMenuInfo(array $menu_links): array {
    // Return defaults when no menu links are present.
    if (empty($menu_links)) {
      return [
        'start_depth' => t('N/A'),
        'max_depth' => t('N/A'),
        'has_children' => FALSE,
      ];
    }

    $menu_depth = [];
    $has_children = FALSE;
    foreach ($menu_links as $menu_link) {
      if (!in_array($menu_link->depth, $menu_depth)) {
        $menu_depth[] = $menu_link->depth;
      }

      if ($menu_link->has_children == 1) {
        $has_children = TRUE;
      }
    }

    sort($menu_depth);

    return [
      'start_depth' => $menu_depth[0],
      'max_depth' => array_pop($menu_depth),
      'has_children' => ($has_children) ? t('Yes') : t('No'),
    ];
  }

  /**
   * Aggregates information from blocks to provide block details related to the menu.
   *
   * @param array $blocks
   *   An array of block objects.
   * @param array $menu
   *   The menu array.
   *
   * @return array
   *   A structured array containing titles and themes where the menu is used.
   */
  private function aggregateBlockInfo(array $blocks, array $menu): array {
    if (!empty($blocks)) {
      $enabled_themes = [];
      $custom_titles = [];
      foreach ($blocks as $block) {

        if ($block->status == "1") {
          if (!empty(drupal_get_path('theme', $block->theme))) {
            if (system_list('theme')[$block->theme]->status == '1') {
              $enabled_themes[] = $block->theme;
            }
          }
        }

        if (!empty($block->title)) {
          if ($block->title != '<none>' ) {
            $custom_titles[$block->theme] = $block->title;
          }
        }

      }
    }

    if (empty($enabled_themes)) {

      // Check for primary/secondary menu links.
      $main_links_source = variable_get('menu_main_links_source', 'main-menu');
      $secondary_links_source = variable_get('menu_secondary_links_source', 'user-menu');

      if ($menu['menu_name'] == $main_links_source) {
        $enabled_themes[] = t('Source for main links.');
      }

      if ($menu['menu_name'] == $secondary_links_source) {
        $enabled_themes[] = t('Source for secondary links.');
      }

      // Check for context module implementation.
      if (module_exists('context')) {
        $contexts = context_load();
        $enabled_contexts = [];

        if (!empty($contexts)) {
          foreach ($contexts as $context) {
            if (isset($context->disabled) && $context->disabled == FALSE) {
              $enabled_contexts[] = $context;
            }
          }
        }

        if (!empty($enabled_contexts)) {
          foreach ($enabled_contexts as $enabled_context) {
            if (!empty($enabled_context->reactions['block'])) {
              if (!empty($enabled_context->reactions['block']['blocks'])) {
                foreach ($enabled_context->reactions['block']['blocks'] as $reaction_block) {
                  if ($reaction_block['delta'] == $menu['menu_name']) {
                    $enabled_themes[] = "Context: $enabled_context->name";
                  }
                }
              }
            }
          }
        }
      }
    }

    return [
      'title' =>  (!empty($custom_titles)) ? $custom_titles : $menu["title"],
      'enabled_themes' =>  (!empty($enabled_themes)) ? $enabled_themes : 'Menu block disabled in block config.',
    ];
  }

  /**
   * Extracts custom menu definitions from module install files.
   *
   * This function scans the .install files of enabled modules to find custom
   * menu definitions. It's used to determine which module defines which menu,
   * aiding in auditing menu usage across the site.
   *
   * @param array $menus
   *   An array of menu machine names to check against module-defined menus.
   *
   * @return array
   *   An associative array mapping module names to their defined menus.
   */
  private function menusFromModules(array $menus = []): array {
    $module_menus = [];

    // Fetch data for all enabled modules.
    $modules = system_rebuild_module_data();
    $enabled_modules = array_filter($modules, function($module) {
      return !empty($module->status);
    });

    foreach ($enabled_modules as $module) {
      $module_dir = drupal_get_path('module', $module->name);

      // Scan .install files for menu definitions/inserts.
      $install_file = "{$module_dir}/{$module->name}.install";
      if (file_exists($install_file)) {
        $file_contents = file_get_contents($install_file);
        if ($file_contents !== false) {

          // Pattern to match menu definitions within the .install file.
          $pattern = "/menu_name'\s*=>\s*'([^']+)'/";

          if (preg_match_all($pattern, $file_contents, $matches)) {
            foreach ($matches[1] as $menu_name) {

              // Only include menus that are being checked.
              if (in_array($menu_name, $menus)) {
                $module_menus[$module->name][] = $menu_name;
              }

            }
          }
        } else {
          // Log an error if the file cannot be read.
          watchdog('audit_export', 'Failed to read file: %file', array('%file' => $install_file), WATCHDOG_ERROR);
        }
      }
    }

    // Ensure unique menu names for each module.
    foreach ($module_menus as &$menus) {
      $menus = array_unique($menus);
    }

    return $module_menus;
  }

}
