<?php

class AuditExportAuditReport {
  /**
   * The machine name value from the AuditExportAudit object.
   * This can be AuditExportAudit or an array.
   *
   * @var mixed
   */
  public $audit;

  /**
   * Data object for the audit report.
   * This can be AuditExportAuditData or an array.
   *
   * @var mixed
   */
  public $data;

  /**
   * The user ID of the author of the report.
   *
   * @var int|null
   */
  public $author = null;

  /**
   * The id of the file object for the report.
   *
   * @var string
   */
  public $fid;

  /**
   * The timestamp of when the report was created.
   *
   * @var int|null
   */
  public $date = null;

  /**
   * Should all audits should be processed?
   *
   * @var bool
   */
  public $process_all = FALSE;

  /**
   * Constructor
   *
   * Note: Typed properties and union types are removed for PHP 7.4 compatibility.
   * Types should be checked manually if needed.
   *
   * @param mixed $audit The AuditExportAudit object or array.
   * @param mixed $data The AuditExportAuditData object or array.
   * @param int|null $author The user ID of the author.
   * @param int|null $date The timestamp when the report was created.
   * @param bool|null $process_all true/false if all audits should be processed.
   */
  public function __construct($audit = [], $data = [], ?int $author = null, ?int $date = null, ?bool $process_all = FALSE) {

    if ($audit === NULL) {
      watchdog('audit_export', 'Audit data is null.');
      return;
    }

    // Skip audit if dependencies are not met.
    if (is_array($audit)) {
      foreach ($audit as $audit_item) {
        if (!$audit_item->checkDependencies()) {
          watchdog('audit_export', t('Dependencies not met for audit @audit.', ['@audit' => $audit_item->name]));
        }
      }
    } else {
      if (!$audit->checkDependencies()) {
        watchdog('audit_export', t('Dependencies not met for audit @audit.', ['@audit' => $audit->name]));
      }
    }

    $this->audit = $audit;
    $this->data = $data;
    $this->author = $author;
    $this->date = $date;
    $this->process_all = $process_all;
  }

  private function setAudit(AuditExportAudit $audit) {
    $this->audit = $audit;
  }

  private function setData(AuditExportAuditData $audit_data) {
    $this->data = $audit_data;
  }

  /**
   * Retrieves the last report for a given audit.
   *
   * @param AuditExportAudit $audit The name of the audit.
   *
   * @return array The last report data or a message with a run report button.
   */
  public function getLastReport(AuditExportAudit $audit): array {
    // Skip audit if dependencies are not met.
    if (!$audit->checkDependencies()) {
      watchdog('audit_export', t('Dependencies not met for audit @audit.', ['@audit' => $audit->name]));
      return [];
    }

    $query = db_select('audit_export_report', 'aer')
      ->fields('aer')
      ->condition('audit', $audit->name)
      ->orderBy('date', 'DESC')
      ->range(0, 1)
      ->execute();

    $record = $query->fetchAssoc();

    if ($record) {
      return $record;
    } else {
      // Return a render array with a message and a button
      return [];
    }
  }

  /**
   * @param $auditName
   *
   * @return array
   * @throws \Exception
   */
  public function processAudit($auditName) {
    $audit = AuditExportAudit::getAudits()[$auditName];
    if (!empty($audit)) {
      if (class_exists($audit->className)) {
        $auditDataInstance = new $audit->className();
        if (method_exists($auditDataInstance, 'prepareData')) {
          return $auditDataInstance->prepareData();
        } else {
          watchdog('audit_export', t("Prepare method for $auditDataInstance not found."));
        }
      }
    }
  }

  /**
   * @return void
   */
  public function reportInit() {
    $audit_names = [];

    // Check for an existing record
    if ($this->process_all) {
      foreach ($this->audit as $audit) {
        $audit_names[] = $audit->name;
      }

    } else {
      $audit_names[] = $this->audit->name;
    }

    foreach ($audit_names as $audit_name) {
      $exists = db_select('audit_export_report', 'aer')
        ->fields('aer', array('audit'))
        ->condition('audit', $audit_name)
        ->execute()
        ->fetchField();

      if ($exists) {
        // If a record exists, reset the 'data' field to an empty array
        db_update('audit_export_report')
          ->fields(array('data' => serialize([]))) // Reset data to an empty serialized array
          ->condition('audit', $audit_name)
          ->execute();
      }
    }
  }

  /**
   * Implement logic to run an audit report
   *
   * @param $audit_list
   * @param bool $cli
   *
   * @return void
   * @throws \Exception
   */
  public function runReport($audit_list, bool $cli = FALSE) {
    // Initialize the report and clear existing data
    if (!$this->process_all) {
      if (!$this->audit->checkDependencies()) {
        watchdog('audit_export', t('Dependencies not met for audit @audit.', ['@audit' => $this->audit->name]));
        return;
      }
    }

    $this->reportInit();
    $records = [];

    // Define batch operations based on records
    $batch = [
      'title' => t('Processing Audit Report.'),
      'operations' => [],
      'init_message' => t('Initialization'),
      'progress_message' => t('Processed @current out of @total.'),
      'error_message' => t('An error occurred during processing'),
      'finished' => "AuditExportAuditReport::auditReportFinished",
    ];

    // Build report main data records.
    if ($this->process_all && is_array($audit_list)) {
      $return_url = '/admin/reports/audit-export/reports';
      foreach ($audit_list as $audit) {

        // Check for dependencies.
        if (!$audit->checkDependencies()) {
          watchdog('audit_export', t('Dependencies not met for audit @audit.', ['@audit' => $audit->name]));
          return;
        }

        // Add CrossTab Scripts.
        $crossTab_scripts = $this->buildCrossTabScriptRecords($audit);
        if (!empty($crossTab_scripts)) {
          foreach ($crossTab_scripts as $crossTab_script) {
            $batch['operations'][] = $crossTab_script;
          }
        }

        // Add dataPreProcess Operations.
        $dataPreProcessOperations = $this->buildDataPreProcessRecords($this, $audit);
        if (!empty($dataPreProcessOperations)) {
          foreach ($dataPreProcessOperations as $dataPreProcessOperation) {
            $batch['operations'][] = $dataPreProcessOperation;
          }
        }

        // Add main data operations
        $records[] = [
          'audit' => $audit,
          'report_ids' => $this->processAudit($audit->name),
          'report_class' => $audit->className,
        ];
      }
    } else {
      // Add CrossTab Scripts
      $crossTab_scripts = $this->buildCrossTabScriptRecords($this->audit);
      if (!empty($crossTab_scripts)) {
        foreach ($crossTab_scripts as $crossTab_script) {
          $batch['operations'][] = $crossTab_script;
        }
      }

      // Build main data operations
      $dataPreProcessOperations = $this->buildDataPreProcessRecords($this, $this->audit);
      if (!empty($dataPreProcessOperations)) {
        foreach ($dataPreProcessOperations as $dataPreProcessOperation) {
          $batch['operations'][] = $dataPreProcessOperation;
          $record_item['report'] = $this;
        }
      }

      // Add main data operations
      $group = $this->audit->group;
      $audit_name = $this->audit->name;
      $return_url = "/admin/reports/audit-export/reports/$group/$audit_name";
      $records = $this->processAudit($audit_list);
    }

    // Add main data operations
    foreach ($records as $record) {
      $record_item['row_data'] = $record;
      $record_item['report'] = $this;
      $batch['operations'][] = ['AuditExportAuditReport::processRecord', [$record_item]];
    }

    batch_set($batch);

    if ($cli) {
      drush_backend_batch_process();
    } else {
      batch_process($return_url);
    }
  }

  /**
   * @param AuditExportAudit $audit
   *
   * @return array
   */
  private function buildCrossTabScriptRecords(AuditExportAudit $audit): array {
    $auditDataInstance = new $audit->className;
    $crossTab_scripts = [];
    $crossTabOperations = [];
    if ($audit->data_type == 'cross') {
      if (method_exists($auditDataInstance, 'prepareCrossTabScripts')) {
        foreach ($auditDataInstance->prepareCrossTabScripts() as $id => $script) {
          $crossTab_scripts[$id] = $script;
        }
      }
    }

    foreach ($crossTab_scripts as $script_info) {
      $crossTabOperations[] = [
        'AuditExportAuditReport::processCrossTabScript',
        [[
          'row_data' => $script_info,
          'report' => $this,
          'audit' => $audit,
          'audit_data' => $auditDataInstance
        ]]
      ];
    }
    return $crossTabOperations;
  }

  /**
   * @param AuditExportAuditReport $report
   *
   * @param AuditExportAudit $audit
   * @param bool $queue
   *
   * @return array
   */
  public function buildDataPreProcessRecords(AuditExportAuditReport $report, AuditExportAudit $audit, bool $queue = FALSE): array {
    $dataPreProcessOperations = [];
    $auditDataInstance = new $audit->className;
    if (!empty($auditDataInstance->processDataPreProcess())) {
      foreach ($auditDataInstance->processDataPreProcess() as $preProcess) {
        if (method_exists($auditDataInstance, $preProcess['method'])) {
          if ($queue) {
            return [
              'row_data' => $preProcess,
              'report' => $this,
              'audit' => $audit,
              'audit_data' => $auditDataInstance
            ];
          }

          $dataPreProcessOperations[] = [
            'AuditExportAuditReport::processPreprocessOperation',
            [[
              'row_data' => $preProcess,
              'report' => $this,
              'audit' => $audit,
              'audit_data' => $auditDataInstance
            ]],
          ];
        }
      }
    }
    return $dataPreProcessOperations;
  }

  /**
   * Logic to process an individual record
   *
   * @param $record
   * @param $context
   *
   * @return void
   */
  public static function processRecord($record, &$context) {
    if ($record["report"]->process_all) {
      $record["report"]->setAudit($record["row_data"]["audit"]);
      $context['message'] = t('Processing @label', array('@label' => $record["row_data"]["audit"]->label));
      $context['results']['report'] = $record['report'];
      foreach ($record['row_data']['report_ids'] as $record_id) {
        if (class_exists($record["row_data"]["report_class"])) {
          $data_class = new $record["row_data"]["report_class"];
          $record["report"]->setData($data_class);
          $processed_record = $record["report"]->data->processData(['row_data' => $record_id]);
          $record["report"]->saveReport($record["report"]->audit->name, $processed_record);
          $context['results']['processed'][] = $record["row_data"]["report_ids"];
        }
      }
    } else {
      if (is_array($record["row_data"])) {
        if (!empty($record["row_data"]["label"])) {
          $process_item_display = $record["row_data"]["label"];
        } else {
          $identifier = $record["report"]->audit->identifier;
          $process_item_display = $record["row_data"][$identifier];
        }
      } else {
        $process_item_display = $record["row_data"];
      }
      $context['message'] = t('Processing @id', array('@id' => $process_item_display));
      $context['results']['report'] = $record['report'];
      $processed_record = $record["report"]->data->processData($record);
      $record["report"]->saveReport($record["report"]->audit->name, $processed_record);
      $context['results']['processed'][] = $record["row_data"];
    }

  }

  /**
   * Logic to process a cross-tabulation script.
   *
   * @param $script_info
   * @param &$context
   *
   * @return void
   */
  public static function processCrossTabScript($script_info, &$context) {
    // Assuming $script_info contains the 'report' object and 'row_data' with script details
    $report = $script_info['report'];
    $script_details = $script_info['row_data'];

    // Dynamically call the prepare method based on script details
    if (class_exists($script_info["audit"]->className)) {
      $data_class_instance = new $script_info["audit"]->className;

      // Verify prepareCrossTabScripts method exists and process scripts.
      if (method_exists($data_class_instance, 'prepareCrossTabScripts')) {
        $crossTabScripts = $data_class_instance->prepareCrossTabScripts();
        foreach ($crossTabScripts as $id => $script) {
          if (is_object($data_class_instance) && method_exists($data_class_instance, $script["method"])) {
            $results = call_user_func([$data_class_instance, $script["method"]], $script_info["row_data"]);
            $report->saveReport($script_info["audit"]->name, $results);
          } else {
            watchdog('audit_export', 'Attempted to call a non-existent method or on a non-object.');
          }
          $context['message'] = t('Processing CrossTab Script @id', ['@id' => $id]);
          $context['results']['processed'][] = $id;
        }
      } else {
        watchdog('audit_export', t("prepareCrossTabScripts method not found for @class.", ['@class' => $script_details['report_class']]));
      }
    } else {
      watchdog('audit_export', t("@class does not exist.", ['@class' => $script_details['report_class']]));
    }
  }

  /**
   * @param $operation
   * @param $context
   *
   * @return void
   */
  public static function processPreprocessOperation($operation, $context) {
    // Assuming $script_info contains the 'report' object and 'row_data' with script details
    $report = $operation['report'];
    $operation_details = $operation['row_data'];
    $data_instance = $operation['audit_data'];
    $audit = $operation['audit'];

    if (method_exists($data_instance, $operation_details["method"])) {
      $results = call_user_func([$data_instance, $operation_details["method"]]);
      $report->saveReport($audit->name, $results);
    }
    $context['message'] = t('Processing CrossTab Script @id', ['@id' => $operation["row_data"]["name"]]);
    $context['results']['processed'][] = $operation["row_data"]["name"];
  }


  /**
   * @param $success
   * @param $results
   * @param $operations
   *
   * @return void
   */
  public static function auditReportFinished($success, $results, $operations) {
    if ($success && !empty($results)) {
      if ($results["report"]->process_all) {
        try {
          $audits = AuditExportAudit::getAudits();
          foreach ($audits as $audit) {
            $results["report"]->saveAuditCSV($audit);
            if (module_exists('audit_export_post')) {
              $remote_post = new AuditExportRemotePost($audit);
              if ($remote_post->isEnabled()) {
                $remote_post->postSend();
              }
            }
          }
        } catch (Exception $e) {
          watchdog('audit_export', 'There was a problem saving the CSV files: '. print_r($e), 'error');
        }
        drupal_set_message(t('Audits complete.'));
      } else {
        if (module_exists('audit_export_post')) {
          $remote_post = new AuditExportRemotePost($results["report"]->audit);
          if ($remote_post->isEnabled()) {
            $remote_post->postSend();
          }
        }
        $results["report"]->saveAuditCSV($results["report"]->audit);
        drupal_set_message(t('Audit complete.'));
      }
    }
    elseif (empty($results)) {
      watchdog('audit_export', 'No results data found for audit.', 'error');
    }
    else {
      drupal_set_message(t('Failed to save the report.'), 'error');
    }
  }

  /**
   * Saves the report to the database and creates a CSV file in the filesystem.
   *
   * @param $audit_name
   * @param $newRecord
   * @param bool $queue
   * @param \AuditExportAuditReport|null $report
   *
   * @return void
   * @throws \Exception
   */
  public function saveReport($audit_name, $newRecord, bool $queue = FALSE, AuditExportAuditReport $report = NULL) {
    if (empty($report) && (get_class($this) == 'AuditExportAuditReport')) {
      $report = $this;
    }

    // Check if table exists
    if (!db_table_exists('audit_export_report')) {
      watchdog('audit_export', 'The database table does not exist.', array(), WATCHDOG_ERROR);
      return;
    }

    $date = REQUEST_TIME;

    // Check for an existing record
    $exists = db_select('audit_export_report', 'aer')
      ->fields('aer')
      ->condition('audit', $audit_name)
      ->execute()
      ->fetchField();

    if (!$exists) {
      // Insert new record if it doesn't exist
      $record = array(
        'audit' => $audit_name,
        'author' => $report->author,
        'date' => $date,
        'data' => ($queue) ? serialize($newRecord): serialize([$newRecord]),
      );
      db_insert('audit_export_report')
        ->fields($record)
        ->execute();
    } else {
      // If the record exists, update the date and data fields.
      $existing_data = db_select('audit_export_report', 'aer')
        ->fields('aer', array('data'))
        ->condition('audit', $audit_name)
        ->execute()
        ->fetchField();

      $existing_rows = unserialize($existing_data);
      if (!is_array($existing_rows)) {
        $existing_rows = [];
      }
      $existing_rows[] = $newRecord;

      db_update('audit_export_report')
        ->fields(array(
          'date' => $date,
          'data' => ($queue) ? serialize($newRecord) : serialize($existing_rows),
        ))
        ->condition('audit', $audit_name)
        ->execute();
    }

    if ($queue) {
      if (module_exists('audit_export_post')) {
        $remote_post = new AuditExportRemotePost(AuditExportAudit::getAudits()[$audit_name]);
        if ($remote_post->isEnabled()) {
          $remote_post->postSend();
        }
      }
    }

  }

  /**
   * Prepares the CSV content from headers and data.
   *
   * @param $headers
   * @param $data
   * @return false|string
   */
  protected function prepareCsvContent($headers, $data) {
    $handle = fopen('php://temp', 'r+');
    fputcsv($handle, $headers);

    $data_processed = unserialize($data['data']);

    foreach ($data_processed as $row) {
      // Assuming each row in $data is already an array in the correct format
      fputcsv($handle, $row);
    }

    rewind($handle);
    $csvContent = stream_get_contents($handle);
    fclose($handle);

    return $csvContent;
  }

  /**
   * CSV file creation and deletion logic.
   *
   * @param $audit
   *
   * @return void
   */
  protected function saveAuditCSV($audit) {
    if (variable_get('audit_export_save_filesystem', 0)) {
      if (class_exists($audit->className)) {
        $data_instance = new $audit->className;
        $data = $data_instance->getData($audit);
        $headers = $data_instance->getHeaders();
        $csvContent = $this->prepareCsvContent($headers, $data);

        $filesystem = variable_get('audit_export_filesystem', 'temporary');
        $filesystemPath = $filesystem . '://' . variable_get('audit_export_filesystem_path', 'audit-export');
        $auditPath = $filesystemPath . '/' . drupal_clean_css_identifier($audit->name);

        // Ensure the directory exists and is writable
        file_prepare_directory($auditPath, FILE_CREATE_DIRECTORY | FILE_MODIFY_PERMISSIONS);

        $auditDataInstance = new $audit->className;
        $auditDataInstanceData = $auditDataInstance->getData($audit);

        if (!empty($auditDataInstanceData["fid"])) {
          $existing_file = file_load($auditDataInstanceData["fid"]);
          if (!empty($existing_file)) {
            file_delete($existing_file);
          }
        }

        // Define the CSV file path
        $csvFilePath = $auditPath . "/audit_export-$audit->name-" . date('YmdHis') . '.csv';

        // Save the CSV content to the file
        $file = file_save_data($csvContent, $csvFilePath, FILE_EXISTS_REPLACE);

        // Update the database with the new file ID
        db_update('audit_export_report')
          ->fields(array('fid' => $file->fid))
          ->condition('audit', $audit->name)
          ->execute();
      }
    }
  }

}
