<?php

/**
 * AuditExportRemotePost class for handling remote post operations.
 */
class AuditExportRemotePost {
  /**
   * @var AuditExportAudit
   */
  protected $audit;

  /**
   * @var array
   */
  protected $configuration;

  /**
   * Constructor.
   *
   * @param AuditExportAudit $audit The audit object to be associated with this remote post handler.
   */
  public function __construct(AuditExportAudit $audit) {
    $this->setAudit($audit);
    $this->setConfig();
  }

  /**
   * Set the audit object.
   *
   * @param AuditExportAudit $audit The audit object to be associated with this remote post handler.
   */
  public function setAudit(AuditExportAudit $audit) {
    $this->audit = $audit;
  }

  /**
   * Set configuration from Drupal variables.
   */
  protected function setConfig() {
    $this->configuration = array(
      'enable_remote_post' => variable_get('audit_export_post_enable_remote_post', FALSE),
      'remote_url' => variable_get('audit_export_post_remote_url', ''),
      'authentication_type' => variable_get('audit_export_post_authentication_type', 'none'),
      'username' => variable_get('audit_export_post_username', ''),
      'password' => variable_get('audit_export_post_password', ''),
    );
  }

  /**
   * Set configuration from Drupal variables.
   */
  public function isEnabled(): bool {
    return (bool) variable_get('audit_export_post_enable_remote_post');
  }

  /**
   * Get configuration from Drupal variables.
   */
  public function getRemotePostURL(): string {
    $url = variable_get('audit_export_post_remote_url');

    // Allow other modules to alter the URL.
    drupal_alter('audit_export_post_url', $url);

    return $url;
  }

  /**
   * Get configuration from Drupal variables.
   */
  public function buildPostData(): array {
    $data = [];
    $audit_data = new $this->audit->className;
    $site_name = str_replace(' ', '_', $this->getSiteName());
    $site_name = preg_replace('/[^a-zA-Z0-9_]/', '', $site_name);
    $site_info = [
      'site_label' => $this->getSiteName(),
      'site_name' => strtolower($site_name),
      'cms' => 'Drupal',
      'version' => (defined('VERSION') ? VERSION : NULL),
      'version_major' => (defined('VERSION') ? explode('.', VERSION)[0] : NULL),
      'php_version' => phpversion(),
      'host' => $_SERVER['HTTP_HOST'],
      'server_ip_addr' => (!empty($_SERVER['SERVER_ADDR'])) ? $_SERVER['SERVER_ADDR'] : NULL,
    ];

    // Use drupal_alter to allow other modules to alter the site_info array
    drupal_alter('audit_export_post_site_info', $site_info);

    $data['site_info'] = $site_info;
    $report_headers = [$audit_data->getHeaders()];
    $data['report_data'] = $audit_data->getData($this->audit);
    $report_audit_data = unserialize($data["report_data"]["data"]);
    $merged_report_data = array_merge(
      $report_headers,
      $report_audit_data
    );
    $data["report_data"]["data"] = drupal_json_encode($merged_report_data);

    return $data;
  }



  /**
   * Get custom or default site nae.
   */
  public function getSiteName(): string {
    return variable_get('audit_export_post_site_name', variable_get('site_name'));
  }

  /**
   * Get the configuration.
   *
   * @return array Configuration array.
   */
  public function getConfig(): array {
    return $this->configuration;
  }

  /**
   * Sends a POST request to the configured remote URL with the provided data.
   *
   */
  public function postSend() {
    $remote_url_default = $this->getRemotePostURL();
    $json_data = drupal_json_encode($this->buildPostData());

    try {
      // Ensure the URL is not empty.
      if (empty($remote_url_default)) {
        throw new Exception('Remote URL is not configured.');
      }

      // Set a higher execution time limit
      set_time_limit(300); // todo: make this configurable

      $ch = curl_init($remote_url_default);
      if ($ch === false) {
        throw new Exception('Failed to initialize cURL.');
      }

      curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
      curl_setopt($ch, CURLOPT_POSTFIELDS, $json_data);
      curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
      curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
      curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_2_0);  // Use HTTP/2
      curl_setopt($ch, CURLOPT_TIMEOUT, 300); // 5 minutes
      curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 60); // 1 minute

      // Allow other modules to alter the curl options.
      $curl_options = array(
        // SSL Options
        // CURLOPT_SSL_VERIFYPEER => false, // todo: make this configurable
        // CURLOPT_SSL_VERIFYHOST => false, // todo: make this configurable
      );
      drupal_alter('audit_export_post_curl_options', $curl_options, $ch);

      // Apply the altered curl options.
      foreach ($curl_options as $option => $value) {
        curl_setopt($ch, $option, $value);
      }

      $response = curl_exec($ch);
      if ($response === false) {
        $curl_error = curl_error($ch);
        $curl_errno = curl_errno($ch);
        curl_close($ch);
        //  watchdog('audit_export_post', 'cURL error: @error (Error code: @code)', ['@error' => $curl_error, '@code' => $curl_errno], WATCHDOG_ERROR); // todo: make this configurable
        throw new Exception('cURL error: ' . $curl_error . ' (Error code: ' . $curl_errno . ')');
      }

      curl_close($ch);

      // watchdog('audit_export_post', 'HTTP response code: @code', ['@code' => $httpcode], WATCHDOG_INFO); // todo: make this configurable
      // watchdog('audit_export_post', 'Response from server: @response', ['@response' => $response], WATCHDOG_INFO); // todo: make this configurable

    } catch (Exception $e) {
      // Log the error.
      watchdog('audit_export_post', 'Error in postSend: @message', ['@message' => $e->getMessage()], WATCHDOG_ERROR);
    }
  }

}
