(function ($, Drupal, drupalSettings) {
  "use strict";

  const FORM_SELECTORS = {
    loginForm: "form.user-login, form.user-login-form",
    registerForm: "form.user-register-form, form.user-form",
    currentPassword: "#edit-current-pass",
    newPassword: "#edit-pass-pass1",
    confirmPassword: "#edit-pass-pass2",
    loginPassword: "#edit-pass",
    loginUsername: "#edit-name",
  };

  const encryptData = (data, key) => {
    return data ? CryptoJS.AES.encrypt(data, key).toString() : "";
  };

  const clearErrorFields = () => {
    if ($(FORM_SELECTORS.loginUsername).hasClass("error")) {
      $(FORM_SELECTORS.loginUsername).val("");
    }
  };

  const handleLoginSubmit = (passkey) => {
    const password = $(FORM_SELECTORS.loginPassword).val();
    const username = $(FORM_SELECTORS.loginUsername).val();

    $(FORM_SELECTORS.loginPassword).val(encryptData(password, passkey));
    $(FORM_SELECTORS.loginUsername).val(encryptData(username, passkey));
    $(FORM_SELECTORS.loginUsername).attr("type", "password");
  };

  const handleRegisterSubmit = (event, passkey) => {
    const currentPassword = $(FORM_SELECTORS.currentPassword).val();
    const newPassword = $(FORM_SELECTORS.newPassword).val();
    const confirmPassword = $(FORM_SELECTORS.confirmPassword).val();

    if (newPassword !== confirmPassword) {
      $(FORM_SELECTORS.confirmPassword).addClass("error").focus();
      event.preventDefault();
      return false;
    }

    $(FORM_SELECTORS.currentPassword).val(
      encryptData(currentPassword, passkey)
    );

    if (newPassword) {
      const encryptedPassword = encryptData(newPassword, passkey);
      $(FORM_SELECTORS.newPassword).val(encryptedPassword);
      $(FORM_SELECTORS.confirmPassword).val(encryptedPassword);
    }
  };

  const getFormEncryptionKey = ($form) => {
    const keyField = $form.find('input[name="auth_encrypt_key"]');

    if (keyField.length) {
      const rawKey = keyField.val();
      return CryptoJS.SHA256(rawKey).toString();
    }

    return null;
  };

  Drupal.behaviors.auth_encrypt = {
    attach: function (context) {
      clearErrorFields();

      once("auth-encrypt-login", FORM_SELECTORS.loginForm, context).forEach(
        (form) => {
          $(form).submit(() => {
            const passkey = getFormEncryptionKey($(form));
            if (passkey) {
              handleLoginSubmit(passkey);
            }
          });
        }
      );

      once(
        "auth-encrypt-register",
        FORM_SELECTORS.registerForm,
        context
      ).forEach((form) => {
        $(form).submit((event) => {
          const passkey = getFormEncryptionKey($(form));
          if (passkey) {
            return handleRegisterSubmit(event, passkey);
          }
          return false;
        });
      });
    },
  };
})(jQuery, Drupal, drupalSettings);
