<?php

namespace Drupal\auth_encrypt\Utility;

use Psr\Log\LoggerInterface;

/**
 * Helper service for decrypting CryptoJS AES encrypted strings.
 */
class AuthEncryptHelper {

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  private const SALT_PREFIX = "Salted__";
  private const CIPHER_METHOD = "aes-256-cbc";
  private const HASH_ALGO = 'md5';
  private const KEY_SIZE = 8;
  private const IV_SIZE = 4;
  private const SALT_LENGTH = 8;
  private const PREFIX_LENGTH = 8;

  /**
   * Constructs a new AuthEncryptHelper object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(LoggerInterface $logger) {
    $this->logger = $logger;
  }

  /**
   * Decrypts a CryptoJS AES encrypted string.
   *
   * @param string $passphrase
   *   The encryption passphrase.
   * @param string $encryptedString
   *   Base64 encoded encrypted string.
   *
   * @return string|false
   *   Decrypted string, or FALSE on failure.
   */
  public function cryptoJsAesDecrypt(string $passphrase, string $encryptedString): string|false {
    if (empty($passphrase) || empty($encryptedString)) {
      $this->logger->error('Empty passphrase or encrypted string');
      return FALSE;
    }

    $cipherText = base64_decode($encryptedString, TRUE);
    if ($cipherText === FALSE || !str_starts_with($cipherText, self::SALT_PREFIX)) {
      $this->logger->error('Invalid encrypted string format');
      return FALSE;
    }

    $salt = substr($cipherText, self::PREFIX_LENGTH, self::SALT_LENGTH);
    $keyData = $this->deriveKeyAndIv($passphrase, $salt);

    $decrypted = openssl_decrypt(
      substr($cipherText, self::PREFIX_LENGTH + self::SALT_LENGTH),
      self::CIPHER_METHOD,
      $keyData['key'],
      OPENSSL_RAW_DATA,
      $keyData['iv']
    );

    if ($decrypted === FALSE) {
      $this->logger->error('Decryption failed');
      return FALSE;
    }

    return $decrypted;
  }

  /**
   * Derives encryption key and IV from a password and salt.
   *
   * Uses MD5 for CryptoJS compatibility.
   *
   * @param string $password
   *   The password for key derivation.
   * @param string $salt
   *   The salt value.
   *
   * @return array{key: string, iv: string}
   *   An associative array with the derived 'key' and 'iv'.
   */
  private function deriveKeyAndIv(string $password, string $salt): array {
    $derivedBytes = '';
    $block = '';
    $targetLength = (self::KEY_SIZE + self::IV_SIZE) * 4;

    while (strlen($derivedBytes) < $targetLength) {
      $block = hash(self::HASH_ALGO, $block . $password . $salt, TRUE);
      $derivedBytes .= $block;
    }

    return [
      'key' => substr($derivedBytes, 0, self::KEY_SIZE * 4),
      'iv'  => substr($derivedBytes, self::KEY_SIZE * 4, self::IV_SIZE * 4),
    ];
  }

}
