<?php

namespace Drupal\author_field\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\CronInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Defines a form to uninstall module dependencies.
 *
 * Removes node.author_field from the database
 * to proceed with the uninstallation.
 */
class AuthorFieldUninstallValidatorForm extends ConfigFormBase {
  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager service.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Cron service.
   */
  protected CronInterface $cronManager;

  /**
   * The item list to be deleted.
   */
  protected array $itemsToDelete;

  /**
   * Constructs a new GroupAuthorFieldUninstallValidator.
   *
   * The config factory for the form.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
        EntityTypeManagerInterface $entity_type_manager,
        EntityFieldManagerInterface $entity_field_manager,
        CronInterface $cron_manager) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->cronManager = $cron_manager;
  }

  /**
   * Instantiate services.
   */
  public static function create(ContainerInterface $container): AuthorFieldSettingsForm|ConfigFormBase|static {
    return new static(
          $container->get('config.factory'),
          $container->get('entity_type.manager'),
          $container->get('entity_field.manager'),
          $container->get('cron')
      );
  }

  /**
   * Set up the form ID.
   */
  public function getFormId(): string {
    return 'author_field_uninstall_settings';
  }

  /**
   * Set up the config file name. It is not used.
   */
  protected function getEditableConfigNames(): array {
    return [
      'author_field.uninstall_settings',
    ];
  }

  /**
   * Shows the field list for content types and storage types.
   *
   * That are required to delete the author field.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $weight = 0;
    $form['title'] = [
      '#markup' => $this->t('<div><b>AUTHOR FIELDS</b></div>'),
      '#weight' => $weight++,
    ];
    $form['break1'] = [
      '#markup' => '<hr>',
      '#weight' => $weight++,
    ];

    $entity_type_ids = ['node' => 'node_type'];
    foreach ($entity_type_ids as $field_type_id => $entity_type_id) {
      $contentTypes = $this->entityTypeManager->getStorage($entity_type_id)
        ->loadMultiple();
      foreach ($contentTypes as $contentType) {
        $type = $contentType->id();
        $fields = array_filter(
              $this->entityFieldManager->getFieldDefinitions($field_type_id, $type),
              function ($fieldDefinition) {
                  return $fieldDefinition instanceof FieldConfigInterface;
              }
          );

        // String to search in entity_reference_revisions.
        $searchStr = 'author';
        foreach ($fields as $key => $definition) {
          if ($definition->getType() == 'author_field') {
            $this->itemsToDelete[$type][] = $key;
          }
          if ($definition->getType() == 'entity_reference_revisions' && str_contains($key, $searchStr)) {
            $this->itemsToDelete[$type][] = $key;
          }
        }
      }
    }

    $body = '<br/><ul>';
    foreach ($this->itemsToDelete as $type => $fieldsArray) {
      foreach ($fieldsArray as $fieldName) {
        $body .= "<li> $fieldName ($type) </li>";
      }
    }

    $body .= '</ul>';

    // Create a Markup object for the body content.
    $safeBody = Markup::create($body);

    $form['body'] = [
      '#markup' => $this->t('<div>@body_text<br/>This action cannot be undone.<br/>Make a backup of your database if you want to be able to restore these items.</div>', ['@body_text' => $safeBody]),
      '#weight' => $weight++,
    ];

    $form['actions']['submit_cancel'] = [
      '#type' => 'submit',
      '#weight' => $weight++,
      '#value' => t('Cancel'),
      '#submit' => [[static::class, 'customFormCancel']],
      '#limit_validation_errors' => [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Cancel button on the form.
   */
  public static function customFormCancel(array &$form, FormStateInterface $form_state) {
    $host = \Drupal::request()->getSchemeAndHttpHost();
    $path = $host . '/admin/modules/uninstall';
    $response = new RedirectResponse($path);
    $response->send();
  }

  /**
   * On submit, execute delete operation of the fields shown on form.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $is_deleted = FALSE;
    $entity_type_ids = ['node'];
    foreach ($this->itemsToDelete as $type => $fieldsArray) {
      foreach ($fieldsArray as $fieldName) {
        foreach ($entity_type_ids as $entity_id) {
          $is_deleted = TRUE;
          $field = FieldConfig::loadByName($entity_id, $type, $fieldName);
          $field?->delete();
        }
      }
    }

    if ($is_deleted) {
      // Run cron job to clean up the fields.
      $this->cronManager->run();
    }

    $this->messenger()->addMessage($this->t('All author fields deleted'));

    $host = $this->getRequest()->getSchemeAndHttpHost();
    $path = $host . '/admin/modules/uninstall';
    $response = new RedirectResponse($path);
    $response->send();

    parent::submitForm($form, $form_state);
  }

}
