<?php

declare(strict_types=1);

namespace Drupal\AutoCacheClear;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Script\ScriptEvents;

/**
 * Automatically clears Drupal's cache after updating dependencies.
 */
final class Plugin implements PluginInterface, EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ScriptEvents::POST_UPDATE_CMD => 'clearCache',
    ];
  }

  /**
   * Clears Drupal's caches by calling rebuild.php.
   */
  public function clearCache(Event $event): void {
    assert(PHP_SAPI === 'cli', 'This script must be run from the command line.');

    // The scripts we're about to run need to be run from the core directory.
    $cwd = dirname((new \ReflectionClass('Drupal'))->getFileName(), 2);

    $executor = $event->getComposer()->getLoop()->getProcessExecutor();
    // First, we need to get a token for the rebuild script.
    $output = '';
    $executor->execute([PHP_BINARY, 'scripts/rebuild_token_calculator.sh'], $output, $cwd);

    // Now we need to parse the token, inject it into $_GET, and call the
    // rebuild script.
    $code = sprintf('parse_str("%s", $_GET); require "rebuild.php";', trim($output));
    $executor->execute([PHP_BINARY, '-r', $code], cwd: $cwd);

    $event->getIO()->write('Drupal caches were cleared.');
  }

  /**
   * {@inheritdoc}
   */
  public function activate(Composer $composer, IOInterface $io): void {
    // Nothing to do here.
  }

  /**
   * {@inheritdoc}
   */
  public function deactivate(Composer $composer, IOInterface $io): void {
    // Nothing to do here.
  }

  /**
   * {@inheritdoc}
   */
  public function uninstall(Composer $composer, IOInterface $io): void {
    // Nothing to do here.
  }

}
