<?php

namespace Drupal\auto_config_form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for automatically generating config forms based on a schema.
 */
abstract class AutoConfigFormBase extends ConfigFormBase {

  /**
   * The Typed Config Manager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected $typedConfigManager;

  /**
   * Create the object and inject dependencies.
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $typed_config_manager = $container->get('config.typed');
    if (!$typed_config_manager) {
      throw new \RuntimeException("Failed to load the Typed Config Manager service.");
    }
    /** @var \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager */
    $instance->typedConfigManager = $typed_config_manager;
    return $instance;
  }

  /**
   * Return the schema key for which a config form should be generated.
   */
  abstract protected function getSchemaKey(): string;

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return [$this->getSchemaKey()];
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * Get the current config values.
   */
  protected function getImmutableConfig(): ImmutableConfig {
    return $this->configFactory->get($this->getSchemaKey());
  }

  /**
   * Get editable config values. Always WITHOUT overrides.
   */
  protected function getEditableConfig(): Config {
    return $this->config($this->getSchemaKey());
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $schema = $this->typedConfigManager->getDefinition($this->getSchemaKey());
    foreach ($schema['mapping'] as $key => $definition) {
      if (in_array($key, ['langcode', '_core'])) {
        continue;
      }
      $form[$key] = $this->createFormElementFromConfigSchemaDefinition([$key], $definition);
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * Create a form element from a config schema definition.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   *
   * @returns array
   *    A partial form array.
   */
  protected function createFormElementFromConfigSchemaDefinition(array $key_path, array $definition): array {
    $editableConfig = $this->getEditableConfig();
    $immutableConfig = $this->getImmutableConfig();

    $full_key = implode('.', $key_path);
    $default_value = $editableConfig->get($full_key);
    $overridden_value = $immutableConfig->get($full_key);
    $original_value = $immutableConfig->getOriginal($full_key, FALSE);

    return match ($definition['type']) {
      'string' => $this->buildStringElement(
        $key_path,
        $definition,
        $default_value,
        $overridden_value,
        $original_value,
      ),
      'boolean' => $this->buildBooleanElement(
        $key_path,
        $definition,
        $default_value,
        $overridden_value,
        $original_value,
      ),
      'float', 'integer' => $this->buildNumberElement(
        $key_path,
        $definition,
        $default_value,
        $overridden_value,
        $original_value,
      ),
      'mapping' => $this->buildGroupElement(
        $key_path,
        $definition,
      ),
      default => $this->buildNotImplementedElement(
        $key_path,
        $definition,
        $default_value,
        $overridden_value,
        $original_value,
      ),
    };
  }

  /**
   * Build a form element for a 'string' config value.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   * @param mixed $default_value
   *   The default value when not set by the user and not overridden.
   * @param mixed $overridden_value
   *   The value set by `settings.php`.
   * @param mixed $original_value
   *   The value before the override.
   *
   * @return array
   *   A partial form array.
   */
  protected function buildStringElement(
    array $key_path,
    array $definition,
    mixed $default_value,
    mixed $overridden_value,
    mixed $original_value,
  ): array {
    return [
      '#type' => 'textfield',
      ...$this->getCommonElementAttributes($key_path, $definition, $default_value, $overridden_value, $original_value),
    ];
  }

  /**
   * Build a fieldset for the inner group of config values.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   *
   * @return array
   *   A partial form array.
   */
  protected function buildGroupElement(
    array $key_path,
    array $definition,
  ): array {
    $result = [
      '#type' => 'fieldset',
      '#title' => $definition['title'] ?? $definition['label'] ?? NULL,
      '#tree' => TRUE,
      '#description' => $definition['description'] ?? NULL,
    ];

    foreach ($definition['mapping'] as $child_key => $child_definition) {
      $result[$child_key] = $this->createFormElementFromConfigSchemaDefinition(
        [...$key_path, $child_key],
        $child_definition,
      );
    }

    return $result;
  }

  /**
   * Build a form element for a 'boolean' config value.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   * @param mixed $default_value
   *   The default value when not set by the user and not overridden.
   * @param mixed $overridden_value
   *   The value set by `settings.php`.
   * @param mixed $original_value
   *   The value before the override.
   *
   * @return array
   *   A partial form array.
   */
  protected function buildBooleanElement(
    array $key_path,
    array $definition,
    mixed $default_value,
    mixed $overridden_value,
    mixed $original_value,
  ): array {
    return [
      '#type' => 'checkbox',
      ...$this->getCommonElementAttributes($key_path, $definition, $default_value, $overridden_value, $original_value),
    ];
  }

  /**
   * Build a form element for a 'float' or 'integer' config value.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   * @param mixed $default_value
   *   The default value when not set by the user and not overridden.
   * @param mixed $overridden_value
   *   The value set by `settings.php`.
   * @param mixed $original_value
   *   The value before the override.
   *
   * @return array
   *   A partial form array.
   */
  protected function buildNumberElement(
    array $key_path,
    array $definition,
    mixed $default_value,
    mixed $overridden_value,
    mixed $original_value,
  ): array {
    $element = [
      '#type' => 'number',
      ...$this->getCommonElementAttributes($key_path, $definition, $default_value, $overridden_value, $original_value),
    ];
    foreach ($definition['constraints'] ?? [] as $name => $value) {
      switch ($name) {
        case 'Range':
          $element['#min'] = $value['min'];
          $element['#max'] = $value['max'];
          break;
      }
    }
    return $element;
  }

  /**
   * In lieu of a form element, just show the details.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   * @param mixed $default_value
   *   The default value when not set by the user and not overridden.
   * @param mixed $overridden_value
   *   The value set by `settings.php`.
   * @param mixed $original_value
   *   The value before the override.
   *
   * @return array
   *   A partial form array.
   */
  protected function buildNotImplementedElement(
    array $key_path,
    array $definition,
    mixed $default_value,
    mixed $overridden_value,
    mixed $original_value,
  ): array {
    return [
      '#type' => 'container',
      0 => [
        '#type' => 'markup',
        '#markup' => $this->t(
          "The config value <code>@key</code> is of type <code>@type</code>, which is not yet implemented.",
          [
            '@key' => implode('.', $key_path),
            '@type' => $definition['type'],
          ]
        ),
        '#prefix' => '<p>',
        '#suffix' => '</p>',
      ],
      1 => [
        '#type' => 'markup',
        '#markup' => $this->t(
          "Please submit a feature request at <a href='@href'>@href</a>",
          [
            '@href' => 'https://www.drupal.org/project/auto_config_form',
          ]
        ),
        '#prefix' => '<p>',
        '#suffix' => '</p>',
      ],
      2 => [
        '#type' => 'markup',
        '#markup' => '<dl>'
        . '<dt>Default value:</dt><dd>' . htmlspecialchars((string) $default_value) . '</dd>'
        . '<dt>Original value:</dt><dd>' . htmlspecialchars((string) $original_value) . '</dd>'
        . '<dt>Overridden value:</dt><dd>' . htmlspecialchars((string) $overridden_value) . '</dd>'
        . '</dl>',
      ],
    ];
  }

  /**
   * Get attributes that are common to all form elements.
   *
   * @param string[] $key_path
   *   The key path of the config value.
   * @param array $definition
   *   The schema definition for the config value.
   * @param mixed $default_value
   *   The default value when not set by the user and not overridden.
   * @param mixed $overridden_value
   *   The value set by `settings.php`.
   * @param mixed $original_value
   *   The value before the override.
   *
   * @return array
   *   A partial form array.
   */
  protected function getCommonElementAttributes(
    array $key_path,
    array $definition,
    mixed $default_value,
    mixed $overridden_value,
    mixed $original_value,
  ): array {
    $element = [
      // @todo In the next major version, use only the new behavior,
      //   as described in
      //   https://www.drupal.org/project/auto_config_form/issues/3515438 .
      '#title' => $definition['title'] ?? $definition['label'],
      '#default_value' => $default_value,
      '#description' => implode("<br>", array_filter([
        $definition['description'] ?? (isset($definition['title']) ? $definition['label'] : ""),
        $this->getOverriddenMessage($overridden_value, $original_value),
      ])),
      '#config_target' => $this->getSchemaKey() . ':' . implode('.', $key_path),
    ];
    foreach ($definition['constraints'] ?? [] as $name => $value) {
      switch ($name) {
        case 'Length':
          $element['#maxlength'] = $value['max'];
          break;

        case 'NotBlank':
        case 'NotNull':
          $element['#required'] = TRUE;
          break;
      }
    }
    return $element;
  }

  /**
   * Build a form element containing the actions (e.g. submit button).
   */
  protected function buildActionsElement(): array {
    return [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save configuration'),
        '#button_type' => 'primary',
      ],
    ];
  }

  /**
   * Get a human-readable override message.
   *
   * @param mixed $overridden
   *   The value after being overridden.
   * @param mixed $original
   *   The value before being overridden.
   *
   * @return string
   *   A human-readable message describing how and why the config value is
   *   overridden.
   */
  protected function getOverriddenMessage(mixed $overridden, mixed $original): string {
    return $overridden === $original ? "" : $this->t(
        "This config value is overridden as: <code>@overridden</code>",
        ['@overridden' => $overridden],
      );
  }

}
