<?php

namespace Drupal\auto_increment_alter\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush command file for AUTO_INCREMENT alter operations.
 */
final class AutoIncrementAlterCommands extends DrushCommands {

  /**
   * Constructs an AutoIncrementAlterCommands object.
   */
  public function __construct(
    private readonly AutoIncrementAlterInterface $autoIncrementAlter,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('auto_increment_alter.mysql'),
    );
  }

  /**
   * Alter the AUTO_INCREMENT value for a single table.
   */
  #[CLI\Command(name: 'auto-increment-alter:table', aliases: ['auto-increment-alter-table', 'aia-t'])]
  #[CLI\Argument(name: 'table', description: 'The name of the table to alter.')]
  #[CLI\Argument(name: 'value', description: 'The new AUTO_INCREMENT value.')]
  #[CLI\Usage(name: 'auto-increment-alter:table node 500', description: 'Alters the node table\'s AUTO_INCREMENT value to 500.')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function alterTable(string $table, int $value): void {
    $this->autoIncrementAlter->alterTable($table, $value);
  }

  /**
   * Alter the AUTO_INCREMENT value for multiple tables.
   */
  #[CLI\Command(name: 'auto-increment-alter:tables', aliases: ['auto-increment-alter-tables', 'aia-ts'])]
  #[CLI\Usage(name: 'auto-increment-alter:tables', description: 'Alter the AUTO_INCREMENT value for all the tables specified in $settings["auto_increment_alter_tables"].')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function alterTables(): void {
    $this->autoIncrementAlter->alterTables();
  }

  /**
   * Alter the AUTO_INCREMENT value for a single content entity.
   */
  #[CLI\Command(name: 'auto-increment-alter:content-entity', aliases: ['auto-increment-alter-content-entity', 'aia-ce'])]
  #[CLI\Argument(name: 'entity_name', description: 'The machine name of the content entity to alter.')]
  #[CLI\Argument(name: 'base_table_value', description: 'The new AUTO_INCREMENT value for the base table.')]
  #[CLI\Argument(name: 'revision_table_value', description: 'The new AUTO_INCREMENT value for the revision table.')]
  #[CLI\Usage(name: 'auto-increment-alter:content-entity node 500', description: 'Alter the AUTO_INCREMENT value of the node entity\'s base table and revision table (if present) to 500.')]
  #[CLI\Usage(name: 'auto-increment-alter:content-entity node 500 1000', description: 'Alter the AUTO_INCREMENT value of the node entity\'s base table to 500 and revision table (if present) to 1000.')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function alterContentEntity(string $entity_name, int $base_table_value, ?int $revision_table_value = NULL): void {
    $this->autoIncrementAlter->alterContentEntity($entity_name, $base_table_value, $revision_table_value);
  }

  /**
   * Alter the AUTO_INCREMENT value for multiple content entities.
   */
  #[CLI\Command(
    name: 'auto-increment-alter:content-entities',
    aliases: [
      'auto-increment-alter-content-entities',
      'aia-ces',
    ],
  )]
  #[CLI\Usage(name: 'auto-increment-alter:content-entities', description: 'Alter the AUTO_INCREMENT value for content entities base/revision tables specified in $settings["auto_increment_alter_content_entities"].')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function alterContentEntities(): void {
    $this->autoIncrementAlter->alterContentEntities();
  }

  /**
   * Get the AUTO_INCREMENT value for a single table.
   */
  #[CLI\Command(name: 'auto-increment-alter:value', aliases: ['auto-increment-alter-value', 'aia-v'])]
  #[CLI\Argument(name: 'table', description: 'The name of the table from which to get AUTO_INCREMENT value')]
  #[CLI\Usage(name: 'auto-increment-alter:value node', description: 'Get the AUTO_INCREMENT value of the node table.')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function getTableAutoIncrementValue(string $table): void {
    $this->autoIncrementAlter->getTableAutoIncrementValue($table);
  }

  /**
   * Get the AUTO_INCREMENT value for multiple tables.
   */
  #[CLI\Command(name: 'auto-increment-alter:values', aliases: ['auto-increment-alter-values', 'aia-vs'])]
  #[CLI\Option(name: 'all', description: 'Returns all tables irrespective of AUTO INCREMENT value.')]
  #[CLI\Usage(name: 'auto-increment-alter:values', description: 'Only return tables that have an AUTO_INCREMENT value set.')]
  #[CLI\Usage(name: 'auto-increment-alter:values --all', description: 'Returns all tables irrespective of AUTO INCREMENT value.')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function getTableAutoIncrementValues(array $options = ['all' => FALSE, 'format' => 'table']): RowsOfFields {
    $rows = [];
    foreach ($this->autoIncrementAlter->getTableAutoIncrementValues($options['all']) as $table_name => $value) {
      $rows[] = [
        'Table name' => $table_name,
        'AUTO_INCREMENT value' => $value,
      ];
    }

    return new RowsOfFields($rows);
  }

  /**
   * Get the list of entities, optionally filtered by group.
   */
  #[CLI\Command(name: 'auto-increment-alter:entity-list', aliases: ['auto-increment-alter-entity-list', 'aia-el'])]
  #[CLI\Option(name: 'group', description: 'If present, only entities that belong to the specified group are returned. Possible values include "content" and "configuration".')]
  #[CLI\Usage(name: 'auto-increment-alter:entity-list', description: 'Get the list of all entities')]
  #[CLI\Usage(name: 'auto-increment-alter:entity-list --group=content', description: 'Get the list of content entities')]
  #[CLI\Usage(name: 'auto-increment-alter:entity-list --group=configuration', description: 'Get the list of configuration entities')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function getEntityList(array $options = ['group' => NULL, 'format' => 'table']): RowsOfFields {
    $rows = [];
    foreach ($this->autoIncrementAlter->getEntityList($options['group']) as $entity_name => $entity_type_object) {
      $rows[] = [
        'Entity name' => $entity_name,
        'Group' => $entity_type_object->getGroup(),
      ];
    }

    return new RowsOfFields($rows);
  }

  /**
   * Get the list of tables.
   */
  #[CLI\Command(name: 'auto-increment-alter:table-list', aliases: ['auto-increment-alter-table-list', 'aia-tl'])]
  #[CLI\Usage(name: 'auto-increment-alter:table-list', description: 'Get the list of tables.')]
  #[CLI\ValidateModulesEnabled(modules: ['mysql'])]
  public function getTableList(array $options = ['format' => 'table']): RowsOfFields {
    $rows = [];
    foreach ($this->autoIncrementAlter->getTableList() as $table_name) {
      $rows[] = [
        'Table name' => $table_name,
      ];
    }

    return new RowsOfFields($rows);
  }

}
