<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter\Form;

use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for setting AUTO_INCREMENT value for a single table.
 */
final class AutoIncrementAlterTableForm extends FormBase {

  /**
   * Constructs an AutoIncrementAlterTableForm object.
   */
  public function __construct(
    protected readonly AutoIncrementAlterInterface $autoIncrementAlterService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_increment_alter.mysql')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_increment_alter_table_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $tableList = $this->autoIncrementAlterService->getTableList();
    $tableOptions = array_combine($tableList, $tableList);

    $request = $this->getRequest();
    $preselectedTable = $request->query->get('name');
    $defaultTableValue = NULL;
    $defaultAutoIncrementValue = NULL;

    if ($preselectedTable && in_array($preselectedTable, $tableList, TRUE)) {
      $defaultTableValue = $preselectedTable;
      $currentValue = $this->autoIncrementAlterService->getTableAutoIncrementValue($preselectedTable);
      if ($currentValue !== NULL) {
        $defaultAutoIncrementValue = (int) $currentValue;
      }
    }

    $form['instructions'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Alter the AUTO_INCREMENT value for a single table. You can specify the %name query string parameter to prepopulate the select list.', ['%name' => 'name']) . '</p>',
    ];

    $form['table'] = [
      '#type' => 'select',
      '#title' => $this->t('Table'),
      '#description' => $this->t('Select a table to alter its AUTO_INCREMENT value'),
      '#options' => $tableOptions,
      '#default_value' => $defaultTableValue,
      '#required' => TRUE,
    ];

    $form['auto_increment_value'] = [
      '#type' => 'number',
      '#title' => $this->t('AUTO_INCREMENT value'),
      '#description' => $this->t('Enter the new AUTO_INCREMENT value for the selected table.'),
      '#min' => 1,
      '#default_value' => $defaultAutoIncrementValue,
      '#required' => TRUE,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $tableName = $form_state->getValue('table');
    $autoIncrementValue = (int) $form_state->getValue('auto_increment_value');

    $this->autoIncrementAlterService->alterTable($tableName, $autoIncrementValue);

    $this->messenger()->addStatus(
      $this->t('AUTO_INCREMENT value for table @table set to @value.', [
        '@table' => $tableName,
        '@value' => $autoIncrementValue,
      ])
    );
  }

}
