<?php

namespace Drupal\auto_login_url;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\user\UserAuthenticationInterface;
use Drupal\user\UserInterface;

/**
 * Service for handling auto login URL authentication.
 */
class AutoLoginUrlLogin {

  /**
   * The config factory.
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * The database connection.
   */
  private Connection $connection;

  /**
   * The general service.
   */
  private AutoLoginUrlGeneral $autoLoginUrlGeneral;

  /**
   * The user authentication service.
   */
  private UserAuthenticationInterface $userAuthentication;

  /**
   * The current user session.
   */
  private AccountProxyInterface $currentUser;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private $logger;

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\auto_login_url\AutoLoginUrlGeneral $autoLoginUrlGeneral
   *   The general service.
   * @param \Drupal\user\UserAuthenticationInterface $userAuthentication
   *   The user authentication service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user session.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    Connection $connection,
    AutoLoginUrlGeneral $autoLoginUrlGeneral,
    UserAuthenticationInterface $userAuthentication,
    AccountProxyInterface $currentUser,
    LoggerChannelFactoryInterface $loggerFactory,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->configFactory = $configFactory;
    $this->connection = $connection;
    $this->autoLoginUrlGeneral = $autoLoginUrlGeneral;
    $this->userAuthentication = $userAuthentication;
    $this->currentUser = $currentUser;
    $this->logger = $loggerFactory->get('auto_login_url');
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Attempts to log in a user with the provided hash.
   *
   * @param int $uid
   *   The user ID.
   * @param string $hash
   *   The authentication hash.
   *
   * @return string|false
   *   The destination URL on success, FALSE on failure.
   */
  public function login(int $uid, string $hash): string|false {
    // Validate inputs.
    if (!$this->validateLoginParameters($uid, $hash)) {
      return FALSE;
    }

    try {
      // Check if hash exists and is valid.
      $login_data = $this->validateAndRetrieveLoginData($uid, $hash);
      if ($login_data === FALSE) {
        return FALSE;
      }

      // Check if token has expired.
      if ($this->isTokenExpired($login_data['timestamp'], $login_data['custom_expiration'] ?? NULL)) {
        $this->logger->warning('Expired auto login token used for user @uid', ['@uid' => $uid]);
        $this->deleteLoginRecord($login_data['id']);
        return FALSE;
      }

      // Optional IP validation (if enabled in config).
      if (!$this->validateIpAddress($login_data)) {
        return FALSE;
      }

      // Load and validate user account.
      $account = $this->loadAndValidateUser($uid);
      if ($account === FALSE) {
        return FALSE;
      }

      // Perform the login.
      $this->performUserLogin($account);

      // Log usage for analytics (if enabled).
      $this->logUrlUsage($login_data);

      // Handle post-login cleanup.
      $this->handlePostLoginCleanup($login_data['id'], $login_data['one_time_use'] ?? NULL);

      // Generate destination URL.
      $destination = $this->generateDestinationUrl($login_data['destination']);

      $this->logger->info('Successful auto login for user @uid to destination @dest', [
        '@uid' => $uid,
        '@dest' => substr($destination, 0, 100),
      ]);

      return $destination;
    }
    catch (\Exception $e) {
      $this->logger->error('Auto login failed for user @uid: @message', [
        '@uid' => $uid,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Validates login parameters.
   *
   * @param int $uid
   *   The user ID.
   * @param string $hash
   *   The hash token.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  private function validateLoginParameters(int $uid, string $hash): bool {
    if (!$this->autoLoginUrlGeneral->validateUserId($uid)) {
      $this->logger->warning('Invalid user ID @uid attempted for auto login', ['@uid' => $uid]);
      return FALSE;
    }

    if (!$this->autoLoginUrlGeneral->validateHashFormat($hash)) {
      $this->logger->warning('Invalid hash format attempted for user @uid', ['@uid' => $uid]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Validates hash and retrieves login data from database.
   */
  private function validateAndRetrieveLoginData(int $uid, string $hash): array|false {
    try {
      // Generate the key for hash verification.
      $auto_login_url_secret = $this->autoLoginUrlGeneral->getSecret();
      $password = $this->autoLoginUrlGeneral->getUserHash($uid);
      $key = Settings::getHashSalt() . $auto_login_url_secret . $password;

      // Generate expected database hash.
      $expected_hash_db = Crypt::hmacBase64($hash, $key);

      // Query database for matching record.
      $result = $this->connection->select('auto_login_url', 'a')
        ->fields('a', ['id', 'uid', 'destination', 'timestamp', 'ip_address', 'custom_expiration', 'one_time_use'])
        ->condition('uid', $uid)
        ->condition('hash', $expected_hash_db)
        ->range(0, 1)
        ->execute()
        ->fetchAssoc();

      if (empty($result)) {
        $this->logger->warning('No matching auto login record found for user @uid', ['@uid' => $uid]);
        return FALSE;
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Database error during login validation: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Validates IP address if IP validation is enabled.
   */
  private function validateIpAddress(array $login_data): bool {
    $config = $this->configFactory->get('auto_login_url.settings');

    // Skip IP validation if not enabled or no IP stored.
    if (!$config->get('validate_ip_address') || empty($login_data['ip_address'])) {
      return TRUE;
    }

    $current_ip = $this->autoLoginUrlGeneral->getClientIp();

    if ($login_data['ip_address'] !== $current_ip) {
      $this->logger->warning('IP address validation failed for auto login. Expected: @expected, Got: @actual, User: @uid', [
        '@expected' => $login_data['ip_address'],
        '@actual' => $current_ip,
        '@uid' => $login_data['uid'],
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Logs URL usage for analytics if enabled.
   */
  private function logUrlUsage(array $login_data): void {
    $config = $this->configFactory->get('auto_login_url.settings');

    // Skip logging if analytics not enabled.
    if (!$config->get('enable_usage_analytics')) {
      return;
    }

    try {
      // Check if analytics table exists.
      if (!$this->connection->schema()->tableExists('auto_login_url_usage')) {
        return;
      }

      // Insert usage record for analytics.
      $this->connection->insert('auto_login_url_usage')
        ->fields([
          'original_id' => $login_data['id'],
          'uid' => $login_data['uid'],
          'used_timestamp' => time(),
          'ip_address' => $this->autoLoginUrlGeneral->getClientIp(),
        ])
        ->execute();
    }
    catch (\Exception $e) {
      // Don't fail login if logging fails.
      $this->logger->warning('Failed to log URL usage: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Checks if a token has expired.
   *
   * @param string $timestamp
   *   The timestamp when the token was created.
   * @param int|null $custom_expiration
   *   Custom expiration time in seconds (NULL = use default setting).
   *
   * @return bool
   *   TRUE if expired, FALSE otherwise.
   */
  private function isTokenExpired(string $timestamp, ?int $custom_expiration): bool {
    // Use custom expiration if provided, otherwise use default.
    if ($custom_expiration !== NULL) {
      $expiration = $custom_expiration;
    }
    else {
      $config = $this->configFactory->get('auto_login_url.settings');
      $expiration = (int) $config->get('expiration');
    }

    return (time() - (int) $timestamp) > $expiration;
  }

  /**
   * Loads and validates a user account.
   */
  private function loadAndValidateUser(int $uid): UserInterface|false {
    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $account = $user_storage->load($uid);

      if (!$account instanceof UserInterface) {
        $this->logger->warning('Failed to load user account @uid', ['@uid' => $uid]);
        return FALSE;
      }

      if ($account->isBlocked()) {
        $this->logger->warning('Attempted auto login for blocked user @uid', ['@uid' => $uid]);
        return FALSE;
      }

      if (!$account->isActive()) {
        $this->logger->warning('Attempted auto login for inactive user @uid', ['@uid' => $uid]);
        return FALSE;
      }

      return $account;
    }
    catch (\Exception $e) {
      $this->logger->error('Error loading user @uid: @message', [
        '@uid' => $uid,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Performs the user login using modern Drupal APIs.
   */
  private function performUserLogin(UserInterface $account): void {
    // Log in the user by switching the current user account.
    $this->currentUser->setAccount($account);

    // Update user's last login timestamp using entity API.
    $account->setLastLoginTime(time());
    $account->save();
  }

  /**
   * Handles post-login cleanup tasks.
   *
   * @param string $record_id
   *   The login record ID.
   * @param int|null $one_time_use
   *   Per-URL one-time use setting (NULL = use default, 0 = no, 1 = yes).
   */
  private function handlePostLoginCleanup(string $record_id, ?int $one_time_use): void {
    // Determine whether to delete based on per-URL or global setting.
    $should_delete = FALSE;

    if ($one_time_use !== NULL) {
      // Use per-URL setting if provided.
      $should_delete = ($one_time_use === 1);
    }
    else {
      // Fall back to global config setting.
      $config = $this->configFactory->get('auto_login_url.settings');
      $should_delete = (bool) $config->get('delete');
    }

    // Delete the login record if configured to do so.
    if ($should_delete) {
      $this->deleteLoginRecord($record_id);
    }
  }

  /**
   * Deletes a login record from the database.
   */
  private function deleteLoginRecord(string $record_id): void {
    try {
      $this->connection->delete('auto_login_url')
        ->condition('id', $record_id)
        ->execute();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete login record @id: @message', [
        '@id' => $record_id,
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Generates the destination URL after login.
   */
  private function generateDestinationUrl(string $destination): string {
    $destination = urldecode($destination);

    // Check if it's already an absolute URL.
    if (str_starts_with($destination, 'http://') || str_starts_with($destination, 'https://')) {
      return $destination;
    }

    // Generate absolute internal URL.
    try {
      // Remove leading slash if present for Url::fromUri.
      $internal_path = ltrim($destination, '/');
      return Url::fromUri('internal:/' . $internal_path, ['absolute' => TRUE])->toString();
    }
    catch (\Exception $e) {
      $this->logger->warning('Failed to generate destination URL for @dest, using front page: @message', [
        '@dest' => $destination,
        '@message' => $e->getMessage(),
      ]);

      // Fallback to front page.
      return Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString();
    }
  }

  /**
   * Cleans up expired tokens from the database.
   *
   * @return int
   *   The number of expired tokens removed.
   */
  public function cleanupExpiredTokens(): int {
    $config = $this->configFactory->get('auto_login_url.settings');
    $expiration = (int) $config->get('expiration');
    $cutoff_time = time() - $expiration;

    try {
      $deleted = $this->connection->delete('auto_login_url')
        ->condition('timestamp', $cutoff_time, '<=')
        ->execute();

      if ($deleted > 0) {
        $this->logger->info('Cleaned up @count expired auto login tokens', ['@count' => $deleted]);
      }

      return $deleted;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to cleanup expired tokens: @message', [
        '@message' => $e->getMessage(),
      ]);
      return 0;
    }
  }

}
