<?php

namespace Drupal\auto_login_url\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for managing auto-login URLs.
 */
class ManageUrlsController extends ControllerBase {

  /**
   * The database connection.
   */
  private Connection $database;

  /**
   * The date formatter.
   */
  private DateFormatterInterface $dateFormatter;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    Connection $database,
    DateFormatterInterface $dateFormatter,
    ConfigFactoryInterface $configFactory,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->database = $database;
    $this->dateFormatter = $dateFormatter;
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Lists all auto-login URLs.
   *
   * @return array
   *   A render array.
   */
  public function listUrls(): array {
    $build = [];

    // Add link to generate new URL.
    $build['generate_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Generate New Auto-Login URL'),
      '#url' => Url::fromRoute('auto_login_url.generate'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
      '#prefix' => '<div class="action-links">',
      '#suffix' => '</div>',
    ];

    // Get configuration.
    $config = $this->configFactory->get('auto_login_url.settings');
    $default_expiration = (int) $config->get('expiration');

    // Query all URLs.
    $query = $this->database->select('auto_login_url', 'a')
      ->fields('a')
      ->orderBy('timestamp', 'DESC')
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(50);

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No auto-login URLs have been created yet.') . '</p>',
      ];
      return $build;
    }

    // Build table.
    $header = [
      $this->t('ID'),
      $this->t('User'),
      $this->t('Destination'),
      $this->t('Created'),
      $this->t('Expires'),
      $this->t('Status'),
      $this->t('One-Time'),
      $this->t('Operations'),
    ];

    $rows = [];
    $current_time = time();
    $user_storage = $this->entityTypeManager->getStorage('user');

    foreach ($results as $record) {
      // Load user.
      $user = $user_storage->load($record->uid);
      $username = $user ? $user->getAccountName() : $this->t('Unknown');

      // Calculate expiration.
      $expiration_seconds = $record->custom_expiration ?? $default_expiration;
      $expires_at = (int) $record->timestamp + $expiration_seconds;
      $is_expired = $current_time > $expires_at;

      // Status badge.
      if ($is_expired) {
        $status = '<span style="color: red; font-weight: bold;">' . $this->t('Expired') . '</span>';
      }
      else {
        $time_remaining = $expires_at - $current_time;
        if ($time_remaining < 3600) {
          $status = '<span style="color: orange;">' . $this->t('Expires soon') . '</span>';
        }
        else {
          $status = '<span style="color: green;">' . $this->t('Active') . '</span>';
        }
      }

      // One-time use indicator.
      $one_time = $record->one_time_use === NULL
        ? $this->t('Default')
        : ($record->one_time_use ? $this->t('Yes') : $this->t('No'));

      // Operations.
      $view_link = Link::createFromRoute(
        $this->t('View'),
        'auto_login_url.view',
        ['id' => $record->id],
        ['attributes' => ['class' => ['button', 'button--primary', 'button--small']]]
      );

      $delete_link = Link::createFromRoute(
        $this->t('Delete'),
        'auto_login_url.delete',
        ['id' => $record->id],
        ['attributes' => ['class' => ['button', 'button--danger', 'button--small']]]
      );

      $operations = [
        'data' => [
          '#markup' => $view_link->toString() . ' ' . $delete_link->toString(),
        ],
      ];

      $rows[] = [
        $record->id,
        $username . ' (' . $record->uid . ')',
        $record->destination,
        $this->dateFormatter->format((int) $record->timestamp, 'short'),
        $this->dateFormatter->format($expires_at, 'short'),
        ['data' => ['#markup' => $status]],
        $one_time,
        $operations,
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No auto-login URLs found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    // Add statistics summary.
    $total_count = $this->database->select('auto_login_url', 'a')
      ->countQuery()
      ->execute()
      ->fetchField();

    $expired_count = $this->database->select('auto_login_url', 'a')
      ->condition('timestamp', $current_time - $default_expiration, '<=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $build['summary'] = [
      '#markup' => '<p><strong>' . $this->t('Total URLs: @total | Active: @active | Expired: @expired', [
        '@total' => $total_count,
        '@active' => $total_count - $expired_count,
        '@expired' => $expired_count,
      ]) . '</strong></p>',
      '#weight' => -10,
    ];

    return $build;
  }

  /**
   * Lists URL usage analytics.
   *
   * @return array
   *   A render array.
   */
  public function listUsage(): array {
    $build = [];

    // Check if analytics table exists.
    if (!$this->database->schema()->tableExists('auto_login_url_usage')) {
      $build['error'] = [
        '#markup' => '<p>' . $this->t('Usage analytics table does not exist. Please enable analytics in the module configuration.') . '</p>',
      ];
      return $build;
    }

    // Query usage data.
    $query = $this->database->select('auto_login_url_usage', 'u')
      ->fields('u')
      ->orderBy('used_timestamp', 'DESC')
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(50);

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No usage data available.') . '</p>',
      ];
      return $build;
    }

    // Build table.
    $header = [
      $this->t('ID'),
      $this->t('Original URL ID'),
      $this->t('User'),
      $this->t('Used At'),
      $this->t('IP Address'),
    ];

    $rows = [];
    $user_storage = $this->entityTypeManager->getStorage('user');

    foreach ($results as $record) {
      // Load user.
      $user = $user_storage->load($record->uid);
      $username = $user ? $user->getAccountName() : $this->t('Unknown');

      $rows[] = [
        $record->id,
        $record->original_id,
        $username . ' (' . $record->uid . ')',
        $this->dateFormatter->format((int) $record->used_timestamp, 'short'),
        $record->ip_address ?? $this->t('N/A'),
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No usage data found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

}
