<?php

namespace Drupal\auto_login_url\Form;

use Drupal\auto_login_url\AutoLoginUrlCreate;
use Drupal\auto_login_url\Exception\AutoLoginUrlException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Password\PasswordInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for administrators to generate auto-login URLs.
 */
class GenerateUrlForm extends FormBase {

  /**
   * The auto login URL create service.
   */
  private AutoLoginUrlCreate $createService;

  /**
   * The current user.
   */
  private AccountProxyInterface $currentUser;

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The password hasher.
   */
  private PasswordInterface $passwordHasher;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * Constructor.
   *
   * @param \Drupal\auto_login_url\AutoLoginUrlCreate $createService
   *   The create service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Password\PasswordInterface $passwordHasher
   *   The password hasher.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    AutoLoginUrlCreate $createService,
    AccountProxyInterface $currentUser,
    EntityTypeManagerInterface $entityTypeManager,
    PasswordInterface $passwordHasher,
    LoggerChannelInterface $logger,
  ) {
    $this->createService = $createService;
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
    $this->passwordHasher = $passwordHasher;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_login_url.create'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('password'),
      $container->get('logger.factory')->get('auto_login_url')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_login_url_generate_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#prefix'] = '<div id="generate-url-form-wrapper">';
    $form['#suffix'] = '</div>';

    // User selection.
    $user_storage = $this->entityTypeManager->getStorage('user');
    $current_user = $user_storage->load($this->currentUser->id());

    $form['user'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('User'),
      '#description' => $this->t('Enter the username or user ID for whom to generate the auto-login URL.'),
      '#target_type' => 'user',
      '#default_value' => $current_user,
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::ajaxUpdateForm',
        'wrapper' => 'generate-url-form-wrapper',
        'event' => 'autocompleteclose change',
      ],
    ];

    $selected_uid = $form_state->getValue('user');
    $current_uid = $this->currentUser->id();

    // Show password field if generating URL for another user.
    if ($selected_uid && $selected_uid != $current_uid) {
      $form['password'] = [
        '#type' => 'password',
        '#title' => $this->t('Your password'),
        '#description' => $this->t('For security, enter your password to generate a URL for another user.'),
        '#required' => TRUE,
      ];
    }

    // Destination URL.
    $form['destination'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Destination'),
      '#description' => $this->t('Enter the destination path (e.g., /user, /node/123) or full URL where the user should be redirected after login.'),
      '#default_value' => '/user',
      '#required' => TRUE,
    ];

    // Advanced settings fieldset.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#description' => $this->t('Override default settings for this specific URL.'),
      '#open' => FALSE,
    ];

    // Custom expiration.
    $form['advanced']['custom_expiration'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom expiration (seconds)'),
      '#description' => $this->t('Leave empty to use the global default. Common values: 300 (5 min), 3600 (1 hour), 86400 (1 day), 2592000 (30 days).'),
      '#min' => 60,
      '#max' => 31536000,
    ];

    // One-time use.
    $form['advanced']['one_time_use'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('One-time use'),
      '#description' => $this->t('If checked, the URL will be deleted after first use (overrides global setting).'),
    ];

    // Absolute URL.
    $form['absolute'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Generate absolute URL'),
      '#description' => $this->t('Include the full domain in the generated URL.'),
      '#default_value' => TRUE,
    ];

    // Submit button.
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate URL'),
      '#button_type' => 'primary',
    ];

    // Display generated URL if available.
    $generated_url = $form_state->get('generated_url');
    if ($generated_url) {
      $form['result'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Generated Auto-Login URL'),
        '#weight' => 100,
      ];

      $form['result']['url'] = [
        '#type' => 'textarea',
        '#title' => $this->t('URL'),
        '#value' => $generated_url,
        '#rows' => 3,
        '#attributes' => [
          'readonly' => 'readonly',
          'onclick' => 'this.select();',
        ],
      ];

      $form['result']['copy_instruction'] = [
        '#markup' => '<p><strong>' . $this->t('Click the URL above to select it, then copy (Ctrl+C or Cmd+C).') . '</strong></p>',
      ];
    }

    return $form;
  }

  /**
   * AJAX callback to update form when user selection changes.
   */
  public function ajaxUpdateForm(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $selected_uid = $form_state->getValue('user');
    $current_uid = $this->currentUser->id();

    $user_storage = $this->entityTypeManager->getStorage('user');

    // Validate user exists.
    $user = $user_storage->load($selected_uid);
    if (!$user) {
      $form_state->setErrorByName('user', $this->t('Invalid user selected.'));
      return;
    }

    // Validate password if generating URL for another user.
    if ($selected_uid != $current_uid) {
      $password = $form_state->getValue('password');
      if (empty($password)) {
        $form_state->setErrorByName('password', $this->t('Password is required to generate a URL for another user.'));
        return;
      }

      // Verify current user's password.
      $current_user = $user_storage->load($current_uid);
      if (!$this->passwordHasher->check($password, $current_user->getPassword())) {
        $form_state->setErrorByName('password', $this->t('Incorrect password.'));
        return;
      }
    }

    // Validate destination is not empty.
    $destination = trim($form_state->getValue('destination'));
    if (empty($destination)) {
      $form_state->setErrorByName('destination', $this->t('Destination is required.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $uid = (int) $form_state->getValue('user');
    $destination = trim($form_state->getValue('destination'));
    $absolute = (bool) $form_state->getValue('absolute');

    // Get custom settings if provided.
    $custom_expiration = $form_state->getValue('custom_expiration');
    $custom_expiration = !empty($custom_expiration) ? (int) $custom_expiration : NULL;

    $one_time_use = $form_state->getValue('one_time_use');
    $one_time_use = $one_time_use ? TRUE : NULL;

    try {
      // Generate the URL.
      $url = $this->createService->create(
        $uid,
        $destination,
        $absolute,
        $custom_expiration,
        $one_time_use
      );

      // Store the URL in form state to display.
      $form_state->set('generated_url', $url);

      // Show success message.
      $this->messenger()->addStatus($this->t('Auto-login URL generated successfully.'));

      // Rebuild the form to show the generated URL.
      $form_state->setRebuild();
    }
    catch (AutoLoginUrlException $e) {
      $this->messenger()->addError($this->t('Failed to generate auto-login URL: @message', [
        '@message' => $e->getMessage(),
      ]));
      $this->logger->error('Form error: @message', ['@message' => $e->getMessage()]);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Unexpected error: @message', [
        '@message' => $e->getMessage(),
      ]));
      $this->logger->error('Unexpected form error: @message | @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
    }
  }

}
