<?php

namespace Drupal\Tests\auto_login_url\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests AJAX functionality in the Generate URL form.
 *
 * @group auto_login_url
 */
class GenerateUrlFormAjaxTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['auto_login_url', 'user'];

  /**
   * An admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer auto login url',
      'access administration pages',
    ]);

    // Create test user.
    $this->testUser = $this->drupalCreateUser();
  }

  /**
   * Tests password field appears when selecting another user.
   */
  public function testPasswordFieldAjaxBehavior(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Initially, password field should not be visible (current user selected).
    $assert->fieldNotExists('password');

    // Select another user using entity reference autocomplete.
    $autocomplete_field = $page->findField('user');
    $autocomplete_field->setValue($this->testUser->getAccountName());

    // Wait for autocomplete suggestions to appear.
    $assert->waitForElementVisible('css', '.ui-autocomplete', 5000);

    // Find and click the autocomplete suggestion.
    $autocomplete_item = $assert->waitForElementVisible('css', '.ui-autocomplete li:first-child', 5000);
    if ($autocomplete_item) {
      $autocomplete_item->click();

      // Wait for AJAX to complete.
      $assert->assertWaitOnAjaxRequest();

      // Password field should now be visible.
      $assert->fieldExists('password');
      $assert->pageTextContains('For security, enter your password');
    }
    else {
      $this->fail('Autocomplete suggestion not found');
    }
  }

  /**
   * Tests form validation with password field.
   */
  public function testPasswordValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Select another user using autocomplete.
    $autocomplete_field = $page->findField('user');
    $autocomplete_field->setValue($this->testUser->getAccountName());

    // Wait for and click autocomplete suggestion.
    $assert->waitForElementVisible('css', '.ui-autocomplete', 5000);
    $autocomplete_item = $assert->waitForElementVisible('css', '.ui-autocomplete li:first-child', 5000);
    if ($autocomplete_item) {
      $autocomplete_item->click();
      $assert->assertWaitOnAjaxRequest();
    }

    // Fill destination.
    $page->fillField('destination', '/user');

    // Try to submit without password.
    $page->pressButton('Generate URL');
    $assert->waitForText('Password is required', 10);

    // Fill incorrect password.
    $password_field = $assert->waitForField('password', 5000);
    $password_field->setValue('wrong_password');
    $page->pressButton('Generate URL');
    $assert->waitForText('Incorrect password', 10);

    // Fill correct password.
    $password_field = $page->findField('password');
    $password_field->setValue($this->adminUser->passRaw);
    $page->pressButton('Generate URL');

    // Should succeed.
    $assert->waitForText('Auto-login URL generated successfully', 10);
  }

  /**
   * Tests URL generation with AJAX form rebuild.
   */
  public function testCompleteAjaxWorkflow(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Select current user initially (no password needed).
    $page->fillField('destination', '/user');
    $page->pressButton('Generate URL');

    // Wait for form submission.
    $assert->pageTextContains('Auto-login URL generated successfully');
    $assert->fieldExists('url');

    // URL field should be read-only.
    $url_field = $page->findField('url');
    $this->assertTrue($url_field->hasAttribute('readonly'));

    // Generated URL should be displayed.
    $generated_url = $url_field->getValue();
    $this->assertNotEmpty($generated_url);
    $this->assertStringContainsString('/autologinurl/', $generated_url);
  }

  /**
   * Tests advanced settings interaction.
   */
  public function testAdvancedSettingsInteraction(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Advanced settings should exist as a details element.
    $details_element = $assert->elementExists('css', 'details[data-drupal-selector="edit-advanced"]');

    // Click to expand advanced settings.
    $summary = $details_element->find('css', 'summary');
    $summary->click();

    // Wait for details to expand and fields to become visible.
    $assert->waitForElementVisible('css', 'input[name="advanced[custom_expiration]"]', 5000);

    // Custom expiration field should be visible.
    $assert->fieldExists('advanced[custom_expiration]');
    $assert->fieldExists('advanced[one_time_use]');

    // Fill in custom values.
    $page->fillField('destination', '/user');
    $page->fillField('advanced[custom_expiration]', 300);
    $page->checkField('advanced[one_time_use]');

    // Submit form.
    $page->pressButton('Generate URL');

    // Should succeed.
    $assert->waitForText('Auto-login URL generated successfully', 10);
  }

}
