<?php

declare(strict_types=1);

namespace Drupal\Tests\auto_login_url\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;

/**
 * Kernel tests for Auto Login URL login service.
 *
 * @group auto_login_url
 */
final class AutoLoginUrlLoginServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'auto_login_url',
    'system',
    'user',
    'field',
  ];

  /**
   * Test user.
   */
  private ?User $testUser = NULL;

  /**
   * The create service.
   *
   * @var \Drupal\auto_login_url\AutoLoginUrlCreate
   */
  private $createService;

  /**
   * The login service.
   *
   * @var \Drupal\auto_login_url\AutoLoginUrlLogin
   */
  private $loginService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['auto_login_url', 'system', 'user']);
    $this->installSchema('auto_login_url', [
      'auto_login_url',
      'auto_login_url_usage',
    ]);

    // Configure for testing.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('max_urls_per_user_per_hour', 1000)
      ->set('token_length', 64)
      ->set('expiration', 86400)
      ->set('delete', FALSE)
      ->set('validate_ip_address', FALSE)
      ->save();

    $this->testUser = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
      'pass' => 'password123',
    ]);
    $this->testUser->save();

    $this->createService = $this->container->get('auto_login_url.create');
    $this->loginService = $this->container->get('auto_login_url.login');
  }

  /**
   * Tests successful login with valid hash.
   */
  public function testSuccessfulLogin(): void {
    $uid = (int) $this->testUser->id();
    $destination = '/user';

    // Create URL.
    $url = $this->createService->create($uid, $destination, FALSE);
    $this->assertNotEmpty($url);

    // Extract hash from URL.
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // Attempt login.
    $result = $this->loginService->login($uid, $hash);

    $this->assertNotFalse($result);
    // Result may include base URL, so just check it contains the destination.
    $this->assertStringContainsString($destination, $result);
  }

  /**
   * Tests login with invalid hash.
   */
  public function testLoginWithInvalidHash(): void {
    $uid = (int) $this->testUser->id();
    $invalid_hash = 'invalid_hash_12345678';

    $result = $this->loginService->login($uid, $invalid_hash);

    $this->assertFalse($result);
  }

  /**
   * Tests login with expired URL.
   */
  public function testLoginWithExpiredUrl(): void {
    $uid = (int) $this->testUser->id();

    // Set short expiration.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('expiration', 1)
      ->save();

    // Create URL.
    $url = $this->createService->create($uid, '/user', FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // Wait for expiration.
    sleep(2);

    // Attempt login should fail.
    $result = $this->loginService->login($uid, $hash);

    $this->assertFalse($result);
  }

  /**
   * Tests delete-on-use functionality.
   */
  public function testDeleteOnUse(): void {
    // Enable delete on use.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('delete', TRUE)
      ->save();

    $uid = (int) $this->testUser->id();

    // Create URL.
    $url = $this->createService->create($uid, '/user', FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // First login should succeed.
    $result1 = $this->loginService->login($uid, $hash);
    $this->assertNotFalse($result1);

    // Second login should fail (URL deleted).
    $result2 = $this->loginService->login($uid, $hash);
    $this->assertFalse($result2);
  }

  /**
   * Tests cleanup of expired tokens.
   */
  public function testCleanupExpiredTokens(): void {
    $uid = (int) $this->testUser->id();

    // Set short expiration.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('expiration', 1)
      ->save();

    // Create several URLs.
    $this->createService->create($uid, '/user1', FALSE);
    $this->createService->create($uid, '/user2', FALSE);
    $this->createService->create($uid, '/user3', FALSE);

    // Wait for expiration.
    sleep(2);

    // Clean up.
    $deleted = $this->loginService->cleanupExpiredTokens();

    $this->assertGreaterThanOrEqual(3, $deleted);

    // Verify database is cleaned.
    $database = $this->container->get('database');
    $count = $database->select('auto_login_url', 'alu')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(0, $count);
  }

  /**
   * Tests usage analytics tracking.
   */
  public function testUsageAnalytics(): void {
    // Enable analytics.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('enable_usage_analytics', TRUE)
      ->save();

    $uid = (int) $this->testUser->id();

    // Create and use URL.
    $url = $this->createService->create($uid, '/user', FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    $this->loginService->login($uid, $hash);

    // Check usage table.
    $database = $this->container->get('database');
    $count = $database->select('auto_login_url_usage', 'alu')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertGreaterThan(0, $count);
  }

  /**
   * Tests login with wrong user ID.
   */
  public function testLoginWithWrongUserId(): void {
    $uid = (int) $this->testUser->id();

    // Create URL for user.
    $url = $this->createService->create($uid, '/user', FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // Try to use hash with different user ID.
    $result = $this->loginService->login(999, $hash);

    $this->assertFalse($result);
  }

  /**
   * Tests login with disabled user.
   */
  public function testLoginWithDisabledUser(): void {
    $uid = (int) $this->testUser->id();

    // Create URL.
    $url = $this->createService->create($uid, '/user', FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // Disable user.
    $this->testUser->set('status', 0);
    $this->testUser->save();

    // Attempt login should fail.
    $result = $this->loginService->login($uid, $hash);

    $this->assertFalse($result);
  }

  /**
   * Tests per-URL custom expiration setting.
   */
  public function testCustomExpirationSetting(): void {
    $uid = (int) $this->testUser->id();

    // Create URL with 2 second custom expiration.
    $url = $this->createService->create($uid, '/user', FALSE, 2, NULL);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // Immediately should work.
    $result = $this->loginService->login($uid, $hash);
    $this->assertNotFalse($result);

    // Create another URL with 2 second expiration.
    $url2 = $this->createService->create($uid, '/user', FALSE, 2, NULL);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url2, $matches2);
    $hash2 = $matches2[1];

    // Wait for expiration.
    sleep(3);

    // Should fail due to custom expiration.
    $result2 = $this->loginService->login($uid, $hash2);
    $this->assertFalse($result2);
  }

  /**
   * Tests per-URL one-time use setting overrides global.
   */
  public function testPerUrlOneTimeUseSetting(): void {
    // Disable global delete setting.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('delete', FALSE)
      ->save();

    $uid = (int) $this->testUser->id();

    // Create URL with one-time use enabled (overriding global FALSE).
    $url = $this->createService->create($uid, '/user', FALSE, NULL, TRUE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // First login should succeed.
    $result1 = $this->loginService->login($uid, $hash);
    $this->assertNotFalse($result1);

    // Second login should fail (URL was deleted).
    $result2 = $this->loginService->login($uid, $hash);
    $this->assertFalse($result2);
  }

  /**
   * Tests per-URL persistent setting overrides global delete.
   */
  public function testPerUrlPersistentSetting(): void {
    // Enable global delete setting.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('delete', TRUE)
      ->save();

    $uid = (int) $this->testUser->id();

    // Create URL with one-time use disabled (overriding global TRUE).
    $url = $this->createService->create($uid, '/user', FALSE, NULL, FALSE);
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $hash = $matches[1];

    // First login should succeed.
    $result1 = $this->loginService->login($uid, $hash);
    $this->assertNotFalse($result1);

    // Second login should also succeed (URL was NOT deleted).
    $result2 = $this->loginService->login($uid, $hash);
    $this->assertNotFalse($result2);
  }

}
