<?php

namespace Drupal\auto_login_url\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for deleting an auto-login URL.
 */
class DeleteUrlForm extends ConfirmFormBase {

  /**
   * The database connection.
   */
  private Connection $database;

  /**
   * The ID of the URL to delete.
   */
  private int $id;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_login_url_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete auto-login URL #@id?', [
      '@id' => $this->id,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auto_login_url.manage');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone. The auto-login URL will be permanently deleted and will no longer work.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?int $id = NULL): array {
    $this->id = $id;

    // Verify the URL exists.
    $record = $this->database->select('auto_login_url', 'a')
      ->fields('a')
      ->condition('id', $id)
      ->execute()
      ->fetchAssoc();

    if (!$record) {
      $this->messenger()->addError($this->t('Auto-login URL not found.'));
      return $this->redirect('auto_login_url.manage');
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    try {
      $deleted = $this->database->delete('auto_login_url')
        ->condition('id', $this->id)
        ->execute();

      if ($deleted) {
        $this->messenger()->addStatus($this->t('Auto-login URL #@id has been deleted.', [
          '@id' => $this->id,
        ]));
      }
      else {
        $this->messenger()->addWarning($this->t('Auto-login URL #@id was not found or already deleted.', [
          '@id' => $this->id,
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to delete auto-login URL: @message', [
        '@message' => $e->getMessage(),
      ]));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
