<?php

namespace Drupal\auto_login_url\Form;

use Drupal\auto_login_url\AutoLoginUrlLogin;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Bulk operations form for auto-login URLs.
 */
class BulkOperationsForm extends ConfirmFormBase {

  /**
   * The database connection.
   */
  private Connection $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The login service.
   */
  private AutoLoginUrlLogin $loginService;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\auto_login_url\AutoLoginUrlLogin $loginService
   *   The login service.
   */
  public function __construct(
    Connection $database,
    ConfigFactoryInterface $configFactory,
    AutoLoginUrlLogin $loginService,
  ) {
    $this->database = $database;
    $this->configFactory = $configFactory;
    $this->loginService = $loginService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('auto_login_url.login')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_login_url_bulk_operations_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete all expired auto-login URLs?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auto_login_url.manage');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    $config = $this->configFactory->get('auto_login_url.settings');
    $expiration = (int) $config->get('expiration');
    $cutoff_time = time() - $expiration;

    $count = $this->database->select('auto_login_url', 'a')
      ->condition('timestamp', $cutoff_time, '<=')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $this->t('This will permanently delete @count expired auto-login URL(s). This action cannot be undone.', [
      '@count' => $count,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Delete All Expired URLs');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $deleted = $this->loginService->cleanupExpiredTokens();

    $this->messenger()->addStatus($this->t('Deleted @count expired auto-login URL(s).', [
      '@count' => $deleted,
    ]));

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
