<?php

namespace Drupal\auto_login_url\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for viewing individual auto-login URL details.
 */
class ViewUrlController extends ControllerBase {

  /**
   * The database connection.
   */
  private Connection $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The date formatter.
   */
  private DateFormatterInterface $dateFormatter;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   */
  public function __construct(
    Connection $database,
    ConfigFactoryInterface $configFactory,
    EntityTypeManagerInterface $entityTypeManager,
    DateFormatterInterface $dateFormatter,
  ) {
    $this->database = $database;
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
    $this->dateFormatter = $dateFormatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays details for a specific auto-login URL.
   *
   * @param int $id
   *   The auto-login URL record ID.
   *
   * @return array
   *   A render array.
   */
  public function viewUrl(int $id): array {
    // Fetch the record.
    $record = $this->database->select('auto_login_url', 'a')
      ->fields('a')
      ->condition('id', $id)
      ->execute()
      ->fetchAssoc();

    if (!$record) {
      throw new NotFoundHttpException();
    }

    // Load user.
    $user_storage = $this->entityTypeManager->getStorage('user');
    $user = $user_storage->load($record['uid']);
    $username = $user ? $user->getAccountName() : $this->t('Unknown');

    // Reconstruct the full URL using hash_url (the URL token, not the HMAC).
    $route_params = [
      'uid' => $record['uid'],
      'hash' => $record['hash_url'] ?? $record['hash'],
    ];
    $full_url = Url::fromRoute('auto_login_url.login', $route_params, ['absolute' => TRUE])->toString();

    // Calculate expiration.
    $config = $this->configFactory->get('auto_login_url.settings');
    $default_expiration = (int) $config->get('expiration');
    $expiration_seconds = $record['custom_expiration'] ?? $default_expiration;
    $expires_at = (int) $record['timestamp'] + $expiration_seconds;
    $is_expired = time() > $expires_at;

    // Status.
    if ($is_expired) {
      $status = [
        '#markup' => '<span style="color: red; font-weight: bold;">' . $this->t('Expired') . '</span>',
      ];
    }
    else {
      $time_remaining = $expires_at - time();
      if ($time_remaining < 3600) {
        $status = [
          '#markup' => '<span style="color: orange; font-weight: bold;">' . $this->t('Expires Soon') . '</span>',
        ];
      }
      else {
        $status = [
          '#markup' => '<span style="color: green; font-weight: bold;">' . $this->t('Active') . '</span>',
        ];
      }
    }

    // Build page.
    $build = [];

    $build['back_link'] = [
      '#type' => 'link',
      '#title' => $this->t('← Back to Manage URLs'),
      '#url' => Url::fromRoute('auto_login_url.manage'),
      '#attributes' => ['class' => ['button']],
    ];

    $build['title'] = [
      '#markup' => '<h2>' . $this->t('Auto-Login URL Details') . '</h2>',
    ];

    // Details table.
    $rows = [
      [
        ['data' => $this->t('ID'), 'header' => TRUE],
        $record['id'],
      ],
      [
        ['data' => $this->t('User'), 'header' => TRUE],
        $username . ' (' . $record['uid'] . ')',
      ],
      [
        ['data' => $this->t('Destination'), 'header' => TRUE],
        $record['destination'],
      ],
      [
        ['data' => $this->t('Status'), 'header' => TRUE],
        $status,
      ],
      [
        ['data' => $this->t('Created'), 'header' => TRUE],
        $this->dateFormatter->format((int) $record['timestamp'], 'long'),
      ],
      [
        ['data' => $this->t('Expires'), 'header' => TRUE],
        $this->dateFormatter->format($expires_at, 'long'),
      ],
      [
        ['data' => $this->t('Custom Expiration'), 'header' => TRUE],
        $record['custom_expiration'] ? $this->t('@seconds seconds', ['@seconds' => $record['custom_expiration']]) : $this->t('Default'),
      ],
      [
        ['data' => $this->t('One-Time Use'), 'header' => TRUE],
        $record['one_time_use'] === NULL ? $this->t('Default') : ($record['one_time_use'] ? $this->t('Yes') : $this->t('No')),
      ],
      [
        ['data' => $this->t('IP Address'), 'header' => TRUE],
        $record['ip_address'] ?? $this->t('N/A'),
      ],
    ];

    $build['details'] = [
      '#type' => 'table',
      '#rows' => $rows,
      '#attributes' => ['class' => ['auto-login-url-details']],
    ];

    // The actual URL.
    $build['url_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Auto-Login URL'),
    ];

    $build['url_section']['url'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Full URL'),
      '#value' => $full_url,
      '#rows' => 3,
      '#attributes' => [
        'readonly' => 'readonly',
        'onclick' => 'this.select();',
        'style' => 'font-family: monospace; font-size: 14px;',
      ],
    ];

    $build['url_section']['copy_instruction'] = [
      '#markup' => '<p><strong>' . $this->t('Click the URL above to select it, then copy (Ctrl+C or Cmd+C).') . '</strong></p>',
    ];

    // Warning if expired.
    if ($is_expired) {
      $build['url_section']['warning'] = [
        '#markup' => '<div class="messages messages--error"><strong>' . $this->t('Warning: This URL has expired and will not work.') . '</strong></div>',
      ];
    }

    // Operations.
    $build['operations'] = [
      '#type' => 'actions',
    ];

    $build['operations']['delete'] = [
      '#type' => 'link',
      '#title' => $this->t('Delete This URL'),
      '#url' => Url::fromRoute('auto_login_url.delete', ['id' => $id]),
      '#attributes' => ['class' => ['button', 'button--danger']],
    ];

    return $build;
  }

}
