<?php

namespace Drupal\Tests\auto_login_url\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Auto Login URL admin UI.
 *
 * @group auto_login_url
 */
class AdminUiTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['auto_login_url', 'user'];

  /**
   * An admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user with proper permissions.
    $this->adminUser = $this->drupalCreateUser([
      'administer auto login url',
      'access administration pages',
    ]);

    // Create test user.
    $this->testUser = $this->drupalCreateUser();
  }

  /**
   * Tests the settings page exists and is accessible.
   */
  public function testSettingsPageAccess(): void {
    // Anonymous users should not have access.
    $this->drupalGet('/admin/people/autologinurl');
    $this->assertSession()->statusCodeEquals(403);

    // Admin user should have access.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Auto Login URL Settings');
  }

  /**
   * Tests the generate URL form.
   */
  public function testGenerateUrlForm(): void {
    $this->drupalLogin($this->adminUser);

    // Access the generate form.
    $this->drupalGet('/admin/people/autologinurl/generate');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Generate Auto-Login URL');

    // Check form elements exist.
    $this->assertSession()->fieldExists('user');
    $this->assertSession()->fieldExists('destination');
    $this->assertSession()->fieldExists('absolute');
    $this->assertSession()->buttonExists('Generate URL');

    // Submit form with current user.
    $this->submitForm([
      'destination' => '/user',
      'absolute' => TRUE,
    ], 'Generate URL');

    // Check for success message.
    $this->assertSession()->pageTextContains('Auto-login URL generated successfully');

    // Check generated URL is displayed.
    $this->assertSession()->fieldExists('url');
  }

  /**
   * Tests generating URL with custom expiration.
   */
  public function testGenerateUrlWithCustomSettings(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('/admin/people/autologinurl/generate');

    // Submit form with destination.
    $this->submitForm([
      'destination' => '/user',
      'absolute' => TRUE,
    ], 'Generate URL');

    $this->assertSession()->pageTextContains('Auto-login URL generated successfully');
  }

  /**
   * Tests the manage URLs page.
   */
  public function testManageUrlsPage(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test URL first.
    $url = auto_login_url_create((int) $this->testUser->id(), '/user', TRUE);
    $this->assertNotEmpty($url);

    // Access the manage page.
    $this->drupalGet('/admin/people/autologinurl/manage');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Manage Auto-Login URLs');

    // Check for table headers.
    $this->assertSession()->pageTextContains('User');
    $this->assertSession()->pageTextContains('Destination');
    $this->assertSession()->pageTextContains('Created');
    $this->assertSession()->pageTextContains('Expires');
    $this->assertSession()->pageTextContains('Status');
    $this->assertSession()->pageTextContains('Operations');

    // Check statistics summary exists.
    $this->assertSession()->pageTextContains('Total URLs');
  }

  /**
   * Tests the delete URL confirmation.
   */
  public function testDeleteUrl(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test URL.
    auto_login_url_create((int) $this->testUser->id(), '/user', FALSE);

    // Get the URL ID from database.
    $database = \Drupal::database();
    $id = $database->select('auto_login_url', 'a')
      ->fields('a', ['id'])
      ->condition('uid', $this->testUser->id())
      ->orderBy('id', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    $this->assertNotEmpty($id);

    // Access delete confirmation page.
    $this->drupalGet("/admin/people/autologinurl/delete/{$id}");
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');

    // Confirm deletion.
    $this->submitForm([], 'Delete');

    // Check for success message.
    $this->assertSession()->pageTextContains('has been deleted');

    // Verify URL was deleted.
    $count = $database->select('auto_login_url', 'a')
      ->condition('id', $id)
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(0, $count);
  }

  /**
   * Tests bulk delete operations.
   */
  public function testBulkDeleteExpiredUrls(): void {
    $this->drupalLogin($this->adminUser);

    // Create URLs with short expiration.
    auto_login_url_create((int) $this->testUser->id(), '/user', FALSE, 1, NULL);

    // Wait for expiration.
    sleep(2);

    // Access bulk delete page.
    $this->drupalGet('/admin/people/autologinurl/bulk-delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete all expired');

    // Confirm bulk deletion.
    $this->submitForm([], 'Delete All Expired URLs');

    // Check for success message.
    $this->assertSession()->pageTextContains('Deleted');
    $this->assertSession()->pageTextContains('expired auto-login URL');
  }

  /**
   * Tests the usage analytics page.
   */
  public function testUsageAnalyticsPage(): void {
    $this->drupalLogin($this->adminUser);

    // Access usage page.
    $this->drupalGet('/admin/people/autologinurl/usage');
    $this->assertSession()->statusCodeEquals(200);

    // Should show the page even if no data.
    $this->assertSession()->pageTextContains('Auto-Login URL Usage');
  }

  /**
   * Tests tab navigation.
   */
  public function testTabNavigation(): void {
    $this->drupalLogin($this->adminUser);

    // Start at generate page.
    $this->drupalGet('/admin/people/autologinurl/generate');
    $this->assertSession()->statusCodeEquals(200);

    // Navigate to manage page.
    $this->drupalGet('/admin/people/autologinurl/manage');
    $this->assertSession()->statusCodeEquals(200);

    // Navigate to usage page.
    $this->drupalGet('/admin/people/autologinurl/usage');
    $this->assertSession()->statusCodeEquals(200);

    // Navigate to health check.
    $this->drupalGet('/admin/reports/auto-login-url/health');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests permissions.
   */
  public function testPermissions(): void {
    // Create user without admin permission.
    $regularUser = $this->drupalCreateUser();
    $this->drupalLogin($regularUser);

    // All admin pages should be forbidden.
    $this->drupalGet('/admin/people/autologinurl');
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalGet('/admin/people/autologinurl/generate');
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalGet('/admin/people/autologinurl/manage');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests URL generation workflow end-to-end.
   */
  public function testCompleteUrlGenerationWorkflow(): void {
    $this->drupalLogin($this->adminUser);

    // Step 1: Generate a URL via the form.
    $this->drupalGet('/admin/people/autologinurl/generate');
    $this->submitForm([
      'destination' => '/user/profile',
      'absolute' => TRUE,
    ], 'Generate URL');

    $this->assertSession()->pageTextContains('Auto-login URL generated successfully');

    // Step 2: Verify it appears in the manage list.
    $this->drupalGet('/admin/people/autologinurl/manage');
    $this->assertSession()->pageTextContains('/user/profile');

    // Step 3: Verify statistics are updated.
    $this->assertSession()->pageTextContains('Total URLs:');
  }

}
