<?php

namespace Drupal\Tests\auto_login_url\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests AJAX functionality in the Generate URL form.
 *
 * @group auto_login_url
 * @group javascript
 */
class GenerateUrlFormAjaxTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['auto_login_url', 'user'];

  /**
   * An admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer auto login url',
      'access administration pages',
    ]);

    // Create test user.
    $this->testUser = $this->drupalCreateUser();
  }

  /**
   * Tests password field appears when selecting another user.
   */
  public function testPasswordFieldAjaxBehavior(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Initially, password field should not be visible (current user selected).
    $assert->fieldNotExists('password');

    // Select another user using entity reference autocomplete.
    $autocomplete_field = $page->findField('user');
    $this->assertNotNull($autocomplete_field, 'User autocomplete field not found');

    // Clear the field first.
    $autocomplete_field->setValue('');

    // Type the username to trigger autocomplete.
    $autocomplete_field->setValue($this->testUser->getAccountName());

    // Trigger keydown event to ensure autocomplete activates.
    $autocomplete_field->keyDown(' ');
    $autocomplete_field->keyUp(' ');

    // Wait for autocomplete suggestions with longer timeout.
    $autocomplete_visible = $assert->waitForElementVisible('css', '.ui-autocomplete', 10000);
    $this->assertNotNull($autocomplete_visible, 'Autocomplete dropdown did not appear within 10 seconds');

    // Wait a moment for items to populate.
    sleep(1);

    // Find and click the autocomplete suggestion.
    $autocomplete_item = $assert->waitForElementVisible('css', '.ui-autocomplete li.ui-menu-item:first-child', 10000);
    $this->assertNotNull($autocomplete_item, 'Autocomplete suggestion not found in dropdown');

    $autocomplete_item->click();

    // Wait for AJAX to complete.
    $assert->assertWaitOnAjaxRequest();

    // Give form time to rebuild.
    sleep(1);

    // Password field should now be visible.
    $assert->fieldExists('password');
    $assert->pageTextContains('For security, enter your password');
  }

  /**
   * Tests form validation with password field.
   */
  public function testPasswordValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Select another user using autocomplete.
    $autocomplete_field = $page->findField('user');
    $this->assertNotNull($autocomplete_field, 'User autocomplete field not found');

    // Clear and set value.
    $autocomplete_field->setValue('');
    $autocomplete_field->setValue($this->testUser->getAccountName());

    // Trigger keydown to activate autocomplete.
    $autocomplete_field->keyDown(' ');
    $autocomplete_field->keyUp(' ');

    // Wait for and click autocomplete suggestion with longer timeout.
    $assert->waitForElementVisible('css', '.ui-autocomplete', 10000);
    sleep(1);

    $autocomplete_item = $assert->waitForElementVisible('css', '.ui-autocomplete li.ui-menu-item:first-child', 10000);
    $this->assertNotNull($autocomplete_item, 'Autocomplete suggestion not found');

    $autocomplete_item->click();
    $assert->assertWaitOnAjaxRequest();
    sleep(1);

    // Fill destination.
    $page->fillField('destination', '/user');

    // Try to submit without password.
    $page->pressButton('Generate URL');
    $assert->waitForText('Password is required', 10);

    // Re-find password field after form validation (element may be stale).
    $password_field = $page->findField('password');
    $this->assertNotNull($password_field, 'Password field not found after validation error');

    // Fill incorrect password.
    $password_field->setValue('wrong_password');
    $page->pressButton('Generate URL');
    $assert->waitForText('Incorrect password', 10);

    // Re-find password field again after second validation.
    $password_field = $page->findField('password');
    $this->assertNotNull($password_field, 'Password field not found after second validation error');

    // Fill correct password.
    $password_field->setValue($this->adminUser->passRaw);
    $page->pressButton('Generate URL');

    // Should succeed.
    $assert->waitForText('Auto-login URL generated successfully', 10);
  }

  /**
   * Tests URL generation with AJAX form rebuild.
   */
  public function testCompleteAjaxWorkflow(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Select current user initially (no password needed).
    $page->fillField('destination', '/user');
    $page->pressButton('Generate URL');

    // Wait for form submission.
    $assert->pageTextContains('Auto-login URL generated successfully');
    $assert->fieldExists('url');

    // URL field should be read-only.
    $url_field = $page->findField('url');
    $this->assertTrue($url_field->hasAttribute('readonly'));

    // Generated URL should be displayed.
    $generated_url = $url_field->getValue();
    $this->assertNotEmpty($generated_url);
    $this->assertStringContainsString('/autologinurl/', $generated_url);
  }

  /**
   * Tests advanced settings interaction.
   */
  public function testAdvancedSettingsInteraction(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/autologinurl/generate');

    $page = $this->getSession()->getPage();
    $assert = $this->assertSession();

    // Advanced settings should exist as a details element.
    $details_element = $assert->elementExists('css', 'details[data-drupal-selector="edit-advanced"]');
    $this->assertNotNull($details_element, 'Advanced settings details element not found');

    // Click to expand advanced settings.
    $summary = $details_element->find('css', 'summary');
    $this->assertNotNull($summary, 'Advanced settings summary not found');
    $summary->click();

    // Wait for details to expand.
    sleep(1);

    // Wait for fields to become visible with longer timeout.
    $custom_expiration_visible = $assert->waitForElementVisible('css', 'input[name="advanced[custom_expiration]"]', 10000);
    $this->assertNotNull($custom_expiration_visible, 'Custom expiration field did not become visible after expanding details');

    // Verify both fields exist and are visible.
    $assert->fieldExists('advanced[custom_expiration]');
    $assert->fieldExists('advanced[one_time_use]');

    // Fill in custom values.
    $page->fillField('destination', '/user');

    // Re-find the field to avoid stale element reference.
    $expiration_field = $page->findField('advanced[custom_expiration]');
    $this->assertNotNull($expiration_field, 'Could not re-find custom expiration field');
    $expiration_field->setValue('300');

    // Check the one-time use checkbox.
    $one_time_field = $page->findField('advanced[one_time_use]');
    $this->assertNotNull($one_time_field, 'Could not find one-time use checkbox');
    $one_time_field->check();

    // Submit form.
    $page->pressButton('Generate URL');

    // Should succeed.
    $assert->waitForText('Auto-login URL generated successfully', 10);
  }

}
