<?php

declare(strict_types=1);

namespace Drupal\Tests\auto_login_url\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;

/**
 * Kernel tests for Auto Login URL creation service.
 *
 * @group auto_login_url
 */
final class AutoLoginUrlCreateServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'auto_login_url',
    'system',
    'user',
    'field',
  ];

  /**
   * Test user.
   */
  private ?User $testUser = NULL;

  /**
   * The create service.
   *
   * @var \Drupal\auto_login_url\AutoLoginUrlCreate
   */
  private $createService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['auto_login_url', 'system', 'user']);
    $this->installSchema('auto_login_url', [
      'auto_login_url',
      'auto_login_url_usage',
    ]);

    // Configure high rate limits for testing.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('max_urls_per_user_per_hour', 1000)
      ->set('token_length', 64)
      ->set('expiration', 86400)
      ->save();

    $this->testUser = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
      'pass' => 'password123',
    ]);
    $this->testUser->save();

    $this->createService = $this->container->get('auto_login_url.create');
  }

  /**
   * Tests URL creation with front destination.
   */
  public function testCreateWithFrontDestination(): void {
    $url = $this->createService->create(
      (int) $this->testUser->id(),
      '<front>',
      FALSE
    );

    $this->assertNotEmpty($url);
    $this->assertStringContainsString('autologinurl', $url);
    $this->assertStringContainsString((string) $this->testUser->id(), $url);
  }

  /**
   * Tests URL creation with custom destination.
   */
  public function testCreateWithCustomDestination(): void {
    $url = $this->createService->create(
      (int) $this->testUser->id(),
      '/user/' . $this->testUser->id(),
      FALSE
    );

    $this->assertNotEmpty($url);
    $this->assertStringContainsString('autologinurl', $url);
  }

  /**
   * Tests URL creation with absolute URL.
   */
  public function testCreateAbsoluteUrl(): void {
    $url = $this->createService->create(
      (int) $this->testUser->id(),
      '<front>',
      TRUE
    );

    $this->assertNotEmpty($url);
    $this->assertStringContainsString('http', $url);
    $this->assertStringContainsString('autologinurl', $url);
  }

  /**
   * Tests URL storage in database.
   */
  public function testUrlStoredInDatabase(): void {
    $uid = (int) $this->testUser->id();
    $destination = '/user';

    $url = $this->createService->create($uid, $destination, FALSE);

    $this->assertNotEmpty($url);

    // Check database.
    $database = $this->container->get('database');
    $query = $database->select('auto_login_url', 'alu')
      ->fields('alu')
      ->condition('uid', $uid)
      ->orderBy('timestamp', 'DESC')
      ->range(0, 1);

    $record = $query->execute()->fetchAssoc();

    $this->assertNotEmpty($record);
    $this->assertEquals($uid, $record['uid']);
    $this->assertStringContainsString($destination, $record['destination']);
  }

  /**
   * Tests multiple URL creation for same user.
   */
  public function testMultipleUrlCreation(): void {
    $uid = (int) $this->testUser->id();

    $url1 = $this->createService->create($uid, '/user', FALSE);
    $url2 = $this->createService->create($uid, '/admin', FALSE);

    $this->assertNotEmpty($url1);
    $this->assertNotEmpty($url2);
    $this->assertNotEquals($url1, $url2);

    // Check both are in database.
    $database = $this->container->get('database');
    $count = $database->select('auto_login_url', 'alu')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(2, $count);
  }

  /**
   * Tests text conversion functionality.
   */
  public function testConvertText(): void {
    $uid = (int) $this->testUser->id();
    $text = 'Click here: [auto-login-url] to login.';

    $converted = $this->createService->convertText($uid, $text);

    $this->assertNotEmpty($converted);
    // Text conversion may not be implemented yet, so just verify
    // it returns a string.
    $this->assertIsString($converted);
  }

  /**
   * Tests token length configuration.
   */
  public function testTokenLengthConfiguration(): void {
    // Set a specific token length.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('token_length', 32)
      ->save();

    $url = $this->createService->create(
      (int) $this->testUser->id(),
      '<front>',
      FALSE
    );

    $this->assertNotEmpty($url);

    // Extract hash from URL.
    preg_match('/autologinurl\/\d+\/([A-Za-z0-9_-]+)/', $url, $matches);
    $this->assertNotEmpty($matches[1]);

    // Hash should be approximately 32 characters (URL-safe base64).
    $hash_length = strlen($matches[1]);
    $this->assertGreaterThan(30, $hash_length);
    $this->assertLessThan(50, $hash_length);
  }

  /**
   * Tests rate limiting integration.
   */
  public function testRateLimitingIntegration(): void {
    // Set a low rate limit.
    $this->container->get('config.factory')
      ->getEditable('auto_login_url.settings')
      ->set('max_urls_per_user_per_hour', 3)
      ->save();

    $uid = (int) $this->testUser->id();

    // Create URLs up to the limit.
    $url1 = $this->createService->create($uid, '/user', FALSE);
    $url2 = $this->createService->create($uid, '/user', FALSE);
    $url3 = $this->createService->create($uid, '/user', FALSE);

    $this->assertNotEmpty($url1);
    $this->assertNotEmpty($url2);
    $this->assertNotEmpty($url3);

    // Fourth attempt should throw exception.
    $this->expectException(\Exception::class);
    $this->createService->create($uid, '/user', FALSE);
  }

}
