<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_deepl\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use DeepL\Translator;

/**
 * Configure Auto Node Translate Deepl settings for this site.
 */
final class LanguageMapping extends ConfigFormBase {

  /**
   * The deepl translator.
   *
   * @var \DeepL\Translator
   */
  protected $translator;

  /**
   * {@inheritdoc}
   */
  public function __construct(private readonly LanguageManagerInterface $languageManager) {
    $this->initializeTranslator();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_node_translate_deepl_language_mapping';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['auto_node_translate_deepl.language_mapping'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('auto_node_translate_deepl.language_mapping');
    $languages = $this->languageManager->getLanguages();
    $form['source_languages'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Source Language Mappings'),
      '#tree' => TRUE,
    ];
    $srcLanguages = $this->translator->getSourceLanguages();
    foreach ($languages as $id => $language) {
      $form['source_languages'][$id] = [
        '#type' => 'select',
        '#options' => $this->getLanguagesOptions($srcLanguages),
        '#title' => $language->getId(),
        '#default_value' => $config->get('source_' . $id),
      ];
    }
    $form['target_languages'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Target Language Mappings'),
      '#tree' => TRUE,
    ];
    $targetLanguages = $this->translator->getTargetLanguages();
    foreach ($languages as $id => $language) {
      $form['target_languages'][$id] = [
        '#type' => 'select',
        '#options' => $this->getLanguagesOptions($targetLanguages),
        '#title' => $language->getId(),
        '#default_value' => $config->get('target_' . $id),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $config = $this->config('auto_node_translate_deepl.language_mapping');
    $languages = $this->languageManager->getLanguages();
    foreach ($languages as $id => $language) {
      $config->set('source_' . $id, $values['source_languages'][$id])
        ->set('target_' . $id, $values['target_languages'][$id]);
    }
    $config->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Initializes the translator with the api key.
   */
  private function initializeTranslator() {
    $config = $this->config('auto_node_translate_deepl.settings');
    $apiKey = $config->get('apikey');
    $this->translator = new Translator($apiKey);
  }

  /**
   * Gets the languages select options.
   *
   * @param array $languages
   *   An array of DeepL/Language objects.
   *
   * @return array
   *   An array of language options with the language code as the key and the
   *    language name as the value.
   */
  private function getLanguagesOptions(array $languages) : array {
    $options = [];
    array_walk($languages, function ($language) use (&$options) {
        $options[$language->code] = $language->name;
    });
    return $options;
  }

}
