<?php

namespace Drupal\auto_rotate_lite\Plugin\ImageEffect;

use Drupal\Core\Image\ImageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\image\Attribute\ImageEffect;
use Drupal\image\ImageEffectBase;
use Drupal\image\ImageEffectInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManager;

/**
 * Automatically rotates an image based on EXIF orientation or other logic.
 */
#[ImageEffect(
  id: "auto_rotate_lite",
  label: new TranslatableMarkup("Auto Rotate Lite"),
  description: new TranslatableMarkup("Automatically rotates images according to orientation flag set by many phones and digital cameras.")
)]
class AutoRotateLiteImageEffect extends ImageEffectBase implements ImageEffectInterface {

    /**
     * {@inheritdoc}
     */
    public function applyEffect(ImageInterface $image) {
        // Test to see if EXIF is supported by the image format and exif_read_data exists.
        if (function_exists('exif_read_data') && in_array($image->getMimeType(), ['image/jpeg', 'image/tiff'])) {
            $path = $image->getSource();
            $exif_orientation = $this->getImageOrientation($path);
            $degrees = 0;
            switch ($exif_orientation) {
                case 3:
                    $degrees = 180;
                    break;
                case 6:
                    $degrees = 90;
                    break;
                case 8:
                    $degrees = 270;
                    break;
            }
            if ($degrees > 0) {
                if (!$image->rotate($degrees)) {
                    $this->logger->error('Image auto-rotate failed using the %toolkit toolkit on %path (%mimetype, %dimensions)', [
                        '%toolkit' => $image->getToolkitId(),
                        '%path' => $image->getSource(),
                        '%mimetype' => $image->getMimeType(),
                        '%dimensions' => $image->getWidth() . 'x' . $image->getHeight(),
                    ]);
                    return FALSE;
                }
            }
        }
        return TRUE;
    }
    
    /**
     * {@inheritdoc}
     */
    public function transformDimensions(array &$dimensions, $uri) {
        $dimensions['width'] = $dimensions['width'] ? (int) $dimensions['width'] : NULL;
        $dimensions['height'] = $dimensions['height'] ? (int) $dimensions['height'] : NULL;

        // Only proceed if both dimensions are available.
        if ($dimensions['width'] && $dimensions['height'] && function_exists('exif_read_data')) {
            $exif_orientation = $this->getImageOrientation($uri);
            if ($exif_orientation && in_array($exif_orientation, [5, 6, 7, 8])) {
                // Swap width and height for rotated images.
                $tmp = $dimensions['width'];
                $dimensions['width'] = $dimensions['height'];
                $dimensions['height'] = $tmp;
            }
            // If orientation is not found, leave dimensions unchanged.
            return;
        }

        // If dimensions are not available, set both to NULL.
        if (!$dimensions['width'] || !$dimensions['height']) {
            $dimensions['width'] = $dimensions['height'] = NULL;
        }
    }
    
    public function getImageOrientation($uri) {
        // Can't get stream wrapper if URI is without scheme.
        if (!StreamWrapperManager::getScheme($uri)) {
            $path = \Drupal::service('file_system')->realpath($uri);
        }
        else {
            /** @var \Drupal\Core\StreamWrapper\StreamWrapperInterface $wrapper */
            $wrapper = \Drupal::service('stream_wrapper_manager')->getViaUri($uri);
            $path = $wrapper->realpath();
            if (empty($path)) {
                $path = $wrapper->getExternalUrl();
            }
        }
        if (empty($path) || !function_exists('exif_read_data')) {
            return NULL;
        }
        $file_exif = @exif_read_data($path);
        // Ensure that the Orientation key|value exists, otherwise leave.
        if (!is_array($file_exif) || !isset($file_exif['Orientation'])) {
            return NULL;
        }
        return $file_exif['Orientation'];
    }
    
}