(function ($, Drupal, drupalSettings, once) {
  'use strict';

  Drupal.behaviors.autoalt = {
    trackedImages: {},

    finishedWorking: function ($button) {
      const fid = $button.data('file-id');
      if (fid) {
        this.trackedImages[fid] = false;
      }
      $button.parent().find('.ajax-progress').remove();

      if (!(drupalSettings.autoalt && drupalSettings.autoalt.hide_button)) {
        $button.show();
      }

      $button
        .closest('.form-managed-file')
        .find("input[name$='[alt]']")
        .removeAttr('disabled');
    },

    generateForButton: function ($button) {
      if (!$button || !$button.length) return;

      const fid = $button.data('file-id');
      if (!fid) return;

      // Prevent duplicate requests.
      if (this.trackedImages[fid]) {
        return;
      }
      this.trackedImages[fid] = true;

      // UI: show throbber, disable input.
      const throbber = $(
        '<div class="ajax-progress autoalt-throbber--right">' +
          '<div class="ajax-progress__throbber">&nbsp;</div>' +
          '<div class="ajax-progress__message">' + Drupal.t('Generating Alt Text...') + '</div>' +
        '</div>'
      );

      const $container = $button.closest('.autoalt-wrapper').length ? $button.closest('.autoalt-wrapper') : $button.parent();
      $container.append(throbber);

      const $fileWrapper = $button.closest('.form-managed-file, .js-form-managed-file');
      $fileWrapper.find("input[name$='[alt]']").attr('disabled', 'disabled');
      $button.hide();

      // Determine language:
      // 1) try to find a select within the same file wrapper (common name/class patterns)
      // 2) fallback to drupalSettings.defaultLanguage
      let language = '';

      if(drupalSettings.autoalt.lang)
      {
        language = (drupalSettings.autoalt && drupalSettings.autoalt.lang) || '';
      }
      else
      {
        language = (drupalSettings.autoalt && drupalSettings.autoalt.language) || '';
      }

      // AJAX request to backend endpoint
      $.ajax({
        url: Drupal.url('api/autoalt/generate'),
        method: 'POST',
        data: { fid: fid, language: language },

        success: (response) => {
          let altText = null;

          if (response && response.success && response.altText) {
            altText = response.altText;
          } else if (response && response.alt_text) {
            altText = response.alt_text;
          }

          if (altText) {
            altText = altText.replace(/^"|"$/g, '');
            $fileWrapper.find("input[name$='[alt]']").val(altText);
          }

          this.finishedWorking($button);
        },

        error: (xhr) => {
          const messenger = new Drupal.Message();
          if (xhr.responseJSON && xhr.responseJSON.error) {
            messenger.add('Error: ' + xhr.responseJSON.error, { type: 'warning' });
          } else {
            messenger.add(Drupal.t('Could not generate alt text. Please try again later.'), { type: 'warning' });
          }

          this.finishedWorking($button);
        },
      });
    },

    attach: function (context) {
      const settings = drupalSettings.autoalt || {};
      const autoGenerate = settings.auto_generate_on_upload || settings.autoGenerateOnUpload || false;

      // Bind manual button click once.
      once('autoalt-button', '.autoalt-button, .ai-alt-text-generation', context).forEach((el) => {
        const $el = $(el);
        $el.on('click', function (e) {
          e.preventDefault();
          Drupal.behaviors.autoalt.generateForButton($el);
        });
      });

      // Hide button when file input changes.
      once('autoalt-file', 'input[type="file"].js-form-file', context).forEach((el) => {
        $(el).on('change', function () {
          const $wrapper = $(this).closest('.js-form-managed-file');
          const $button = $wrapper.find('.autoalt-button, .ai-alt-text-generation');
          if ($button.length) {
            $button.hide();
          }
        });
      });

      // Initial autogenerate if alt is empty and fid exists.
      if (autoGenerate) {
        $(context).find('.autoalt-button, .ai-alt-text-generation').each(function () {
          const $btn = $(this);
          const fid = $btn.data('file-id') || $btn.attr('data-file-id') || null;
          const $altField = $btn.closest('.form-managed-file').find("input[name$='[alt]']");

          if (fid && $altField.length && !$altField.val()) {
            if (!Drupal.behaviors.autoalt.trackedImages[fid]) {
              Drupal.behaviors.autoalt.generateForButton($btn);
            }
          }
        });
      }

      // Detect Drupal AJAX upload completion.
      $(document).ajaxComplete((event, xhr, settings) => {
        if (!settings || !settings.url) return;

        const uploadDetected = settings.url.includes('/file/ajax/field_') || settings.url.includes('/media/ajax/');

        if (!uploadDetected) return;

        $('.js-form-managed-file').each(function () {
          const $wrapper = $(this);
          const $hiddenFid = $wrapper.find('input[type="hidden"][name$="[fids]"]');
          const $button = $wrapper.find('.autoalt-button, .ai-alt-text-generation');

          if ($hiddenFid.length && $hiddenFid.val() && $button.length) {
            const fid = $hiddenFid.val();
            $button.attr('data-file-id', fid).data('file-id', fid);
            $button.show();

            if (autoGenerate) {
              const $altField = $wrapper.find("input[name$='[alt]']");
              if ($altField.length && !$altField.val()) {
                if (!Drupal.behaviors.autoalt.trackedImages[fid]) {
                  Drupal.behaviors.autoalt.generateForButton($button);
                }
              }
            }
          }
        });
      });
    },
  };
})(jQuery, Drupal, drupalSettings, once);
