<?php

namespace Drupal\autoalt\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\file\Entity\File;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Config\ConfigFactoryInterface;


class AutoaltController extends ControllerBase
{

	protected $httpClient;
	protected $fileSystem;
	protected $configFactory;
	protected $fileUrlGenerator;

	public function __construct(
		Client $http_client,
		FileSystemInterface $file_system,
		ConfigFactoryInterface $config_factory,
		FileUrlGeneratorInterface $file_url_generator
	) {
		$this->httpClient = $http_client;
		$this->fileSystem = $file_system;
		$this->configFactory = $config_factory;
		$this->fileUrlGenerator = $file_url_generator;
	}

	public static function create(ContainerInterface $container)
	{
		return new static(
			$container->get('http_client'),
			$container->get('file_system'),
			$container->get('config.factory'),
			$container->get('file_url_generator')
		);
	}

	public function generate(Request $request)
	{
		try {
			$fid = $request->request->get('fid');
			$language = $request->request->get('language');

			if (!$fid) {
				throw new \InvalidArgumentException('Missing file ID.');
			}

			$file = File::load($fid);
			if (!$file) {
				throw new \InvalidArgumentException('Invalid file ID.');
			}

			$file_uri = $file->getFileUri();
			$file_path = $this->fileSystem->realpath($file_uri);

			if (!$file_path || !file_exists($file_path)) {
				throw new \InvalidArgumentException('File not found.');
			}

			$config = $this->configFactory->get('autoalt.settings');

			$apiKey = $config->get('api_key');

			$filter_ext = $config->get('filter_extensions') ?: [];

			$file_extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));

			if (!empty($filter_ext) && !in_array($file_extension, $filter_ext)) {
				return new JsonResponse([
					'success' => FALSE,
					'error' => "The file extension '.{$file_extension}' is not allowed for generate alt text."
				], 400);
			}

			if (!$apiKey) {
				throw new \InvalidArgumentException('API key not configured.');
			}

			$writingStyle      = $config->get('writing_style') ?: '';
			$prefix            = $config->get('prefix') ?: '';
			$suffix            = $config->get('suffix') ?: '';
			$site_public       = (bool) $config->get('site_publicly_accessible');
			$seo_keywords      = $config->get('seo_keywords') ?: '';
			$negative_keywords = $config->get('negative_keywords') ?: '';
			$custom_prompt     = $config->get('custom_prompt') ?: '';
			$min_length        = (int) $config->get('min_alt_length') ?: 125;
			$max_length        = (int) $config->get('max_alt_length') ?: 150;

			$dataUri = '';
			if ($site_public != 1) {
				$imageData = base64_encode(file_get_contents($file_path));
				$mimeType  = mime_content_type($file_path);
				$dataUri   = "data:{$mimeType};base64,{$imageData}";
			}

			$image_url = $this->fileUrlGenerator->generateAbsoluteString($file_uri);

			$payload = [
				'image_url'              => $image_url,
				'base64_img'             => $dataUri,
				'language'               => $language,
				'ai_writing_style'       => $writingStyle,
				'hardcoded_string_begin' => $prefix,
				'hardcoded_string_end'   => $suffix,
				'seo_keywords'           => $seo_keywords,
				'negative_keywords'      => $negative_keywords,
				'customPromptFromUser'   => $custom_prompt,
				'alttext_min_limit'      => $min_length,
				'alttext_max_limit'      => $max_length,
			];

			$response = $this->httpClient->post('https://ahxdfj.autoalt.ai/api/autoalt-generate-alt', [
				'headers' => [
					'Authorization' => 'Bearer ' . $apiKey,
					'Content-Type'  => 'application/json',
					'Accept'        => 'application/json',
				],
				'json' => $payload,
			]);

			if ($response->getStatusCode() === 200) {
				$result = json_decode($response->getBody()->getContents(), TRUE);
				return new JsonResponse(['success' => TRUE, 'altText' => $result['alt_text']]);
			}

			throw new \RuntimeException('Invalid API response.');
		} catch (GuzzleException $e) {
			\Drupal::logger('autoalt')->error('API error: @error', ['@error' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => 'API error occurred.'], 500);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('Error: @error', ['@error' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 400);
		}
	}
}
