<?php

namespace Drupal\autoalt\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Configure AutoAlt.ai settings.
 */
class AutoaltGeneratorSettingsForm extends ConfigFormBase
{

  protected const WRITING_STYLE = [
    '' => 'Use account default',
    'neutral' => 'Neutral — Clear, balanced, and objective',
    'friendly' => 'Friendly — Warm, approachable, and informal',
    'professional' => 'Professional — Polished and business-appropriate',
    'casual' => 'Casual — Laid-back and conversational',
    'witty' => 'Witty — Clever, playful, with light humor',
    'confident' => 'Confident — Assertive and bold',
    'empathetic' => 'Empathetic — Sensitive and understanding',
    'inspiring' => 'Inspiring — Motivational and uplifting',
    'technical' => 'Technical — Precise, jargon-friendly, and expert-level',
    'minimalist' => 'Minimalist — Simple, clean, and concise',
    'luxury' => 'Luxury — Elegant and premium-sounding',
    'youthful' => 'Youthful — Trendy, fresh, and informal',
    'quirky' => 'Quirky — Fun, offbeat, and creative',
    'sales_oriented' => 'Sales-Oriented — Persuasive and call-to-action driven',
    'informative' => 'Informative — Educational and fact-based',
    'formal' => 'Formal — Structured, respectful, and traditional',
    'authoritative' => 'Authoritative — Expert and trustworthy',
    'seo_optimized' => 'SEO-Optimized — Keyword-rich and search-focused',
  ];

  public function getFormId()
  {
    return 'autoalt_settings';
  }

  protected function getEditableConfigNames()
  {
    return ['autoalt.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('autoalt.settings');
    $apiKey = $config->get('api_key');

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter API Key'),
      '#description' => $this->t('Enter your API key. You can get API key from <a href="@dashboard" target="_blank">AutoAlt.ai</a>.', [
        '@dashboard' => 'https://autoalt.ai/'
      ]),
      '#default_value' => $apiKey,
      '#required' => TRUE,
    ];

    if ($apiKey) {
      try {
        $client = new Client();
        $response = $client->post('https://ahxdfj.autoalt.ai/api/autoalt-credit-count', [
          'headers' => [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
          ],
        ]);

        if ($response->getStatusCode() === 200) {
          $data = json_decode($response->getBody(), TRUE);
          $form['autoalt_credits_info'] = [
            '#type' => 'markup',
            '#markup' => '<div class="credit-info-box" >' . $this->t('Your current credit balance is @credits / @total_credit', [
              '@credits' => $data['credit'] ?? 0,
              '@total_credit' => $data['total_credit'] ?? 0,
            ]) . '</div>',
          ];
        }
      } catch (GuzzleException $e) {
        $form['api_error'] = [
          '#type' => 'markup',
          '#markup' => '<div class="credit-info-box">' . $this->t('Invalid API key. Please check your credentials.') . '</div>',
        ];
        \Drupal::logger('autoalt')->error('Error verifying API key: @error', ['@error' => $e->getMessage()]);
      }
    }

    $form['writing_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Writing Style'),
      '#description' => $this->t('Choose the tone for generating the alt text.'),
      '#options' => self::WRITING_STYLE,
      '#default_value' => $config->get('writing_style') ?: '',
    ];

    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix for Alt Text'),
      '#description' => $this->t('Optional prefix to prepend to generated alt text.'),
      '#default_value' => $config->get('prefix') ?: '',
    ];

    $form['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix for Alt Text'),
      '#description' => $this->t('Optional suffix to append to generated alt text.'),
      '#default_value' => $config->get('suffix') ?: '',
    ];

    $form['filter_extensions'] = [
      '#type' => 'select',
      '#title' => $this->t('Filter by Image Extension'),
      '#description' => $this->t('Only generate alt text for selected image extensions.'),
      '#options' => [
        'jpg'  => 'jpg',
        'jpeg' => 'jpeg',
        'png'  => 'png',
        'gif'  => 'gif',
        'webp' => 'webp',
        'svg'  => 'svg',
      ],
      '#default_value' => $config->get('filter_extensions') ?: [],
      '#multiple' => TRUE,
      '#size' => 6,
    ];

    $form['auto_generate_on_upload'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-Generate on Image Upload'),
      '#default_value' => (bool) $config->get('auto_generate_on_upload') ?: true,
    ];

    $form['site_publicly_accessible'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('This site is publicly accessible on the internet'),
      '#default_value' => (bool) $config->get('site_publicly_accessible') ?: true,
      '#suffix' => "<div id='autoalt-cloudflare-note'>
        <div id='autoalt-cloudflare-note-heading'>
            Notice for Cloudflare or .htpasswd Protected Sites
        </div>
        <p>
            If your website is using <strong>Cloudflare’s “Under Attack Mode”</strong> or is 
            <strong>protected with an .htpasswd,</strong> please  <mark>Unchecked</mark> the checkbox
            <strong>“This site is publicly accessible on the internet.”</strong>
        </p>
    </div>",
    ];

    $form['seo_keywords'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SEO Keywords'),
      '#description' => $this->t('Comma-separated keywords to bias the generated alt text toward SEO.'),
      '#default_value' => $config->get('seo_keywords') ?: '',
    ];

    $form['negative_keywords'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Negative Keywords'),
      '#description' => $this->t('Comma-separated words to avoid in generated alt text.'),
      '#default_value' => $config->get('negative_keywords') ?: '',
    ];

    $form['custom_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt'),
      '#description' => $this->t('A custom prompt to send to the AI for generating alt text.'),
      '#default_value' => $config->get('custom_prompt') ?: '',
    ];

    $form['min_alt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum Alt Text Length'),
      '#default_value' => $config->get('min_alt_length') ?: 100,
      '#description' => $this->t('Minimum number of characters for alt text ( 50 - 100 )'),
      '#min' => 50,
      '#max' => 100,
    ];

    $form['max_alt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Alt Text Length'),
      '#default_value' => $config->get('max_alt_length') ?: 125,
      '#description' => $this->t('Maximum number of characters for alt text ( 100 - 250 )'),
      '#min' => 100,
      '#max' => 250,
    ];

    $form = parent::buildForm($form, $form_state);
    $form['#attached']['library'][] = 'autoalt/admin-settings';
    $form['actions']['submit']['#value'] = $this->t('Save');

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $this->config('autoalt.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('writing_style', $form_state->getValue('writing_style'))
      ->set('prefix', $form_state->getValue('prefix'))
      ->set('suffix', $form_state->getValue('suffix'))
      ->set('filter_extensions', array_filter((array) $form_state->getValue('filter_extensions')))
      ->set('auto_generate_on_upload', (bool) $form_state->getValue('auto_generate_on_upload'))
      ->set('site_publicly_accessible', (bool) $form_state->getValue('site_publicly_accessible'))
      ->set('seo_keywords', $form_state->getValue('seo_keywords'))
      ->set('negative_keywords', $form_state->getValue('negative_keywords'))
      ->set('custom_prompt', $form_state->getValue('custom_prompt'))
      ->set('min_alt_length', (int) $form_state->getValue('min_alt_length'))
      ->set('max_alt_length', (int) $form_state->getValue('max_alt_length'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}
