(function ($, Drupal, once) {
  'use strict';

  console.log('🔥 autoalt-admin-settings.js loaded');

  Drupal.behaviors.autoaltAdminOtp = {
    attach: function (context) {

      /* =========================================================
       * OTP / REGISTER LOGIC (SCOPED TO REGISTER BOX)
       * ========================================================= */
      once('autoalt-admin-otp', '#autoalt-register-box', context).forEach(function (wrapper) {

        const $box = $(wrapper);

        let countdownInterval = null;
        let secondsRemaining = 300; // 5 minutes
        let autoVerifyTriggered = false;

        const $otpWrapper = $box.find('.autoalt-otp-wrapper');
        const $timerText = $box.find('.autoalt-otp-timer span');
        const $verifyBtn = $box.find('input[value="Verify OTP"], button:contains("Verify OTP")');
        const $resendBtn = $box.find('.autoalt-resend-otp');
        const $otpInputs = $box.find('.autoalt-otp-box');

        /* ================= FORMAT TIME ================= */
        const formatTime = (sec) => {
          const m = Math.floor(sec / 60);
          const s = sec % 60;
          return `${String(m).padStart(2, '0')}:${String(s).padStart(2, '0')}`;
        };

        /* ================= STOP TIMER ================= */
        const stopTimer = () => {
          if (countdownInterval) {
            clearInterval(countdownInterval);
            countdownInterval = null;
            console.log('🛑 OTP timer stopped');
          }
        };

        /* ================= START TIMER ================= */
        const startOtpTimer = () => {
          stopTimer();

          secondsRemaining = 300;
          autoVerifyTriggered = false;

          console.log('⏱ Starting OTP timer (5 minutes)');

          $verifyBtn.prop('disabled', false).removeClass('is-disabled');
          $resendBtn.hide();
          $timerText.text(formatTime(secondsRemaining));
          $box.find('.autoalt-otp-expired-msg').remove();

          countdownInterval = setInterval(() => {
            secondsRemaining--;

            if (secondsRemaining <= 0) {
              stopTimer();

              console.warn('⛔ OTP expired');

              $timerText.text('Expired');
              $verifyBtn.prop('disabled', true).addClass('is-disabled');
              $resendBtn.show();

              $('<div class="autoalt-otp-expired-msg" style="color:#d63638;margin-top:8px;">' +
                'The verification code has expired. Please request a new one.' +
                '</div>').insertAfter($timerText);

              return;
            }

            $timerText.text(formatTime(secondsRemaining));
          }, 1000);
        };

        /* ================= AUTO VERIFY CHECK ================= */
        const tryAutoVerify = () => {
          if (autoVerifyTriggered) {
            return;
          }

          let otp = '';
          let filled = true;

          $otpInputs.each(function () {
            const val = $(this).val();
            if (!val || !/^\d$/.test(val)) {
              filled = false;
            }
            otp += val;
          });

          if (filled && otp.length === 6) {
            autoVerifyTriggered = true;
            console.log('✅ OTP complete → auto verify:', otp);

            setTimeout(() => {
              $verifyBtn.trigger('mousedown').trigger('click');
            }, 200);
          }
        };

        /* ================= INPUT HANDLING ================= */
        $otpInputs.each(function (index) {

          $(this).on('input', function () {
            const val = $(this).val();

            if (!/^\d$/.test(val)) {
              $(this).val('');
              return;
            }

            if (index < $otpInputs.length - 1) {
              $otpInputs.eq(index + 1).focus();
            }

            tryAutoVerify();
          });

          $(this).on('keydown', function (e) {
            if (e.key === 'Backspace' && !$(this).val()) {
              if (index > 0) {
                $otpInputs.eq(index - 1).focus();
              }
            }
          });

        });

        /* ================= PASTE HANDLING ================= */
        $otpInputs.on('paste', function (e) {
          e.preventDefault();

          const pasted = (e.originalEvent || e)
            .clipboardData
            .getData('text')
            .replace(/\D/g, '');

          if (pasted.length !== 6) {
            console.warn('⚠ Invalid OTP paste:', pasted);
            return;
          }

          pasted.split('').forEach((digit, index) => {
            $otpInputs.eq(index).val(digit);
          });

          $otpInputs.last().focus();
          tryAutoVerify();
        });

        /* ================= CHECK OTP SENT ================= */
        const checkOtpMessage = () => {
          const $msg = $('.messages--status:contains("Verification code sent")');

          if ($msg.length) {
            console.log('✅ OTP SENT → showing OTP UI');

            $box.find('input[type="email"]').closest('.form-item').hide();
            $box.find('input[value="Send Verification Code"], button:contains("Send Verification Code")').hide();

            $otpWrapper.show();
            startOtpTimer();
          }
        };

        /* ================= RESEND OTP ================= */
        $resendBtn.on('click', function () {
          console.log('🔁 Resend OTP clicked');
          setTimeout(startOtpTimer, 500);
        });

        checkOtpMessage();
        $(document).ajaxComplete(checkOtpMessage);
      });

      /* =========================================================
      * CLEAR API KEY CONFIRM – FINAL FIX
      * ========================================================= */
      once('autoalt-clear-api', '.autoalt-clear-api', context).forEach(function (btn) {
        btn.addEventListener(
          'mousedown',
          function (e) {
            const confirmed = window.confirm(
              'Are you sure you want to clear the API key?'
            );

            if (!confirmed) {
              e.preventDefault();
              e.stopImmediatePropagation();
              return false;
            }
          },
          true 
        );
      });


    }
  };

})(jQuery, Drupal, once);
