<?php

namespace Drupal\autoalt\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\file\Entity\File;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\Exception\ClientException;
use Drupal;


class AutoaltController extends ControllerBase
{

	protected $httpClient;
	protected $fileSystem;
	protected $configFactory;
	protected $fileUrlGenerator;

	public function __construct(
		Client $http_client,
		FileSystemInterface $file_system,
		ConfigFactoryInterface $config_factory,
		FileUrlGeneratorInterface $file_url_generator
	) {
		$this->httpClient = $http_client;
		$this->fileSystem = $file_system;
		$this->configFactory = $config_factory;
		$this->fileUrlGenerator = $file_url_generator;
	}

	public static function create(ContainerInterface $container)
	{
		return new static(
			$container->get('http_client'),
			$container->get('file_system'),
			$container->get('config.factory'),
			$container->get('file_url_generator')
		);
	}

	public function generate(Request $request)
	{
		try {
			$fid               = $request->request->get('fid');
			$language          = $request->request->get('language');
			$saveHistory       = $request->request->get('saveHistory');
			$seo_keywords      = $request->request->get('seo_keywords');
			$negative_keywords = $request->request->get('negative_keywords');

			if (!$fid) {
				throw new \InvalidArgumentException('Missing file ID.');
			}

			$file = File::load($fid);
			if (!$file) {
				throw new \InvalidArgumentException('Invalid file ID.');
			}

			$file_uri = $file->getFileUri();
			$file_path = $this->fileSystem->realpath($file_uri);

			if (!$file_path || !file_exists($file_path)) {
				throw new \InvalidArgumentException('File not found.');
			}

			$config     = $this->configFactory->get('autoalt.settings');
			$apiKey     = $config->get('api_key');
			$filter_ext = $config->get('filter_extensions') ?: [];

			$file_extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));

			if (!empty($filter_ext) && !in_array($file_extension, $filter_ext)) {
				return new JsonResponse([
					'success' => FALSE,
					'error' => "The file extension '.{$file_extension}' is not allowed for generate alt text."
				], 400);
			}

			if (!$apiKey) {
				throw new \InvalidArgumentException('API key not configured.');
			}

			$writingStyle  = $config->get('writing_style') ?: '';
			$prefix        = $config->get('prefix') ?: '';
			$suffix        = $config->get('suffix') ?: '';
			$site_public   = (bool) $config->get('site_publicly_accessible');
			$image_name    = (bool) $config->get('auto_generate_image_name');
			$custom_prompt = $config->get('custom_prompt') ?: '';
			$min_length    = (int) $config->get('min_alt_length') ?: 125;
			$max_length    = (int) $config->get('max_alt_length') ?: 150;

			$dataUri = '';
			if ($site_public != 1) {
				$imageData = base64_encode(file_get_contents($file_path));
				$mimeType  = mime_content_type($file_path);
				$dataUri   = "data:{$mimeType};base64,{$imageData}";
			}

			$image_url = $this->fileUrlGenerator->generateAbsoluteString($file_uri);

			$payload = [
				'image_url'              => $image_url,
				'base64_img'             => $dataUri,
				'language'               => $language,
				'ai_writing_style'       => $writingStyle,
				'hardcoded_string_begin' => $prefix,
				'hardcoded_string_end'   => $suffix,
				'seo_keywords'           => $seo_keywords,
				'negative_keywords'      => $negative_keywords,
				'customPromptFromUser'   => $custom_prompt,
				'alttext_min_limit'      => $min_length,
				'alttext_max_limit'      => $max_length,
				'autoaltai_image_name'   => $image_name ? 'on' : 'off',
			];

			$response = $this->httpClient->post('https://ahxdfj.autoalt.ai/api/autoalt-generate-alt', [
				'headers' => [
					'Authorization' => 'Bearer ' . $apiKey,
					'Content-Type'  => 'application/json',
					'Accept'        => 'application/json',
				],
				'json' => $payload,
			]);

			if ($response->getStatusCode() === 200) {
				$result = json_decode($response->getBody()->getContents(), TRUE);

				if ($saveHistory) {
					$uid = \Drupal::currentUser()->id() ?: 0;
					$now = \Drupal::time()->getRequestTime();

					\Drupal::database()->insert('autoalt_history')
						->fields([
							'fid' => (int) $fid,
							'langcode' => $language,
							'alt_text' => $result['alt_text'],
							'processed_by' => (int) $uid,
							'processed_on' => (int) $now,
							'source' => 'single-generate-alt',
						])->execute();
				}

				return new JsonResponse(['success' => TRUE, 'altText' => $result['alt_text'], 'imagename' => $image_name ? $result['imagename'] : '']);
			}

			throw new \RuntimeException('Invalid API response.');
		} catch (ClientException $e) {

			$resp = $e->getResponse();
			$status = $resp ? $resp->getStatusCode() : 500;
			$body = $resp ? (string) $resp->getBody() : $e->getMessage();

			$data = json_decode($body, true);
			$apiMessage = $data['message'] ?? ($data['error'] ?? $body);

			\Drupal::logger('autoalt')->error('API client error: @msg', ['@msg' => $apiMessage]);

			return new JsonResponse([
				'success' => FALSE,
				'error'   => $apiMessage,
				'raw'     => $body,
			], $status);
		} catch (GuzzleException $e) {
			\Drupal::logger('autoalt')->error('API error: @error', ['@error' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('Error: @error', ['@error' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 400);
		}
	}

	public function saveAlt(Request $request)
	{
		$fid       = $request->request->get('fid');
		$alt       = $request->request->get('alt');
		$language  = $request->request->get('language', NULL);
		$update_id = $request->request->get('update_id');

		if (!$fid || $alt === NULL) {
			return new JsonResponse(['success' => FALSE, 'error' => 'Missing parameters'], 400);
		}

		$imagename = $request->request->get('imagename');
		$imagename = is_string($imagename) ? trim($imagename) : NULL;

		try {
			$media_query = \Drupal::entityQuery('media')
				->accessCheck(FALSE)
				->exists('field_media_image')
				->condition('field_media_image.target_id', (int) $fid);

			if ($language && $language !== 'all') {
				$media_query->condition('langcode', $language);
			}

			$mids = $media_query->execute();
			if (empty($mids)) {
				return new JsonResponse(['success' => FALSE, 'error' => 'No related media found'], 400);
			}

			$updated = 0;
			$medias = \Drupal\media\Entity\Media::loadMultiple($mids);
			$uid = \Drupal::currentUser()->id() ?: 0;
			$now = \Drupal::time()->getRequestTime();

			foreach ($medias as $media) {
				if ($language && $language !== 'all' && $media->hasTranslation($language)) {
					$m = $media->getTranslation($language);
				} else {
					$m = $media;
				}

				if ($m && $m->hasField('field_media_image')) {
					$items = $m->get('field_media_image');
					if (!$items->isEmpty()) {

						foreach ($items as $delta => $field_item) {
							if ((int) $field_item->target_id === (int) $fid) {
								$field_item->alt = $alt;

								if (!empty($imagename)) {
									if ($m->hasField('name')) {
										$m->set('name', $imagename);
									} elseif (method_exists($m, 'setName')) {
										$m->setName($imagename);
									}
								}

								$m->save();
								$updated++;

								if ($update_id) {
									\Drupal::database()->update('autoalt_history')
										->fields([
											'alt_text' => $alt,
											'processed_by' => (int) $uid,
											'processed_on' => (int) $now,
											'source' => 'manual-update',
										])
										->condition('id', (int) $update_id)
										->execute();
								} else {
									\Drupal::database()->insert('autoalt_history')
										->fields([
											'fid' => (int) $fid,
											'mid' => (int) $media->id(),
											'langcode' => $language ?: $m->language()->getId(),
											'alt_text' => $alt,
											'processed_by' => (int) $uid,
											'processed_on' => (int) $now,
											'source' => 'save-alt',
										])->execute();
								}

								break;
							}
						}
					}
				}
			}

			if ($updated > 0) {
				return new JsonResponse(['success' => TRUE, 'updated' => $updated]);
			}
			return new JsonResponse(['success' => FALSE, 'error' => 'No media updated'], 500);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('saveAlt error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}


	public function listFids(Request $request)
	{
		try {
			$limit          = (int) $request->request->get('limit', 0);
			$language       = $request->request->get('language', 'all');
			$skip_processed = (bool) $request->request->get('skip_processed', FALSE);

			$processed_fids = [];
			if ($skip_processed) {
				try {
					$processed_fids = \Drupal::database()
						->select('autoalt_history', 'h')
						->fields('h', ['fid'])
						->distinct()
						->execute()
						->fetchCol();
					if (!is_array($processed_fids)) {
						$processed_fids = [];
					}

					$processed_fids = array_map('intval', $processed_fids);
					$processed_index = array_flip($processed_fids);
				} catch (\Exception $e) {

					\Drupal::logger('autoalt')->warning('Could not fetch processed fids: @e', ['@e' => $e->getMessage()]);
					$processed_fids = [];
					$processed_index = [];
				}
			} else {
				$processed_index = [];
			}

			$mq = \Drupal::entityQuery('media')->accessCheck(FALSE)->exists('field_media_image');

			if ($language && $language !== 'all') {
				$mq->condition('langcode', $language);
			}

			if ($limit > 0) {
				$mq->range(0, $limit);
			}

			$mids = $mq->execute();

			$pairs = [];
			$grouped = [];
			$seen = [];

			if (!empty($mids)) {
				$medias = \Drupal\media\Entity\Media::loadMultiple($mids);

				foreach ($medias as $media) {
					if ($language && $language !== 'all') {
						if (! $media->hasTranslation($language)) {
							continue;
						}
						$m = $media->getTranslation($language);

						if (! $m->hasField('field_media_image')) {
							continue;
						}
						$items = $m->get('field_media_image');
						if ($items->isEmpty()) {
							continue;
						}

						foreach ($items as $field_item) {

							$fid = (int) $field_item->target_id;
							$langcode = $language;
							if ($fid) {
								if ($skip_processed && isset($processed_index[$fid])) {
									continue;
								}

								$key = $fid . '|' . $langcode;
								if (isset($seen[$key])) {
									continue;
								}
								$seen[$key] = TRUE;
								$pairs[] = ['fid' => $fid, 'langcode' => $langcode];
								$grouped[$fid][] = $langcode;
							}
						}
					} else {
						$translations = $media->getTranslationLanguages();

						if (empty($translations)) {
							$base_lang = $media->language()->getId();
							$translations = [$base_lang => NULL];
						}

						foreach ($translations as $langcode => $langData) {
							if (is_object($langcode) && method_exists($langcode, 'getId')) {
								$langcode = $langcode->getId();
							}
							if (! $media->hasTranslation($langcode)) {
								continue;
							}
							$mtrans = $media->getTranslation($langcode);

							if (! $mtrans->hasField('field_media_image')) {
								continue;
							}

							$items = $mtrans->get('field_media_image');
							if ($items->isEmpty()) {
								continue;
							}

							foreach ($items as $field_item) {

								$fid = (int) $field_item->target_id;
								if (! $fid) {
									continue;
								}

								if ($skip_processed && isset($processed_index[$fid])) {
									continue;
								}

								$key = $fid . '|' . $langcode;
								if (isset($seen[$key])) {
									continue;
								}
								$seen[$key] = TRUE;
								$pairs[] = ['fid' => $fid, 'langcode' => $langcode];
								$grouped[$fid][] = $langcode;
							}
						}
					}
				}
			}

			$total_all = count($pairs);

			return new JsonResponse([
				'success' => TRUE,
				'fids' => array_values($pairs),
				'grouped' => $grouped,
				'total' => count($pairs),
				'total_all' => $total_all,
			]);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('ListFids error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}


	public function processedByAutoaltCount(Request $request)
	{
		try {
			$skipProcessed = TRUE;
			$processedIndex = [];

			if ($skipProcessed) {
				try {
					$processedFids = \Drupal::database()
						->select('autoalt_history', 'h')
						->fields('h', ['fid'])
						->distinct()
						->execute()
						->fetchCol() ?: [];

					$processedIndex = array_flip(array_map('intval', $processedFids));
				} catch (\Exception $e) {
					\Drupal::logger('autoalt')->warning('Could not fetch processed fids: @e', ['@e' => $e->getMessage()]);
					$processedIndex = [];
				}
			}

			$mids = \Drupal::entityQuery('media')
				->accessCheck(FALSE)
				->exists('field_media_image')
				->execute();

			$processedByAutoalt = 0;
			$seen = [];

			if (!empty($mids)) {
				$medias = \Drupal\media\Entity\Media::loadMultiple($mids);

				foreach ($medias as $media) {

					$langCodes = array_keys($media->getTranslationLanguages() ?: [$media->language()->getId() => NULL]);

					foreach ($langCodes as $langcode) {
						if (! $media->hasTranslation($langcode)) {
							continue;
						}

						$mtrans = $media->getTranslation($langcode);

						if (! $mtrans->hasField('field_media_image')) {
							continue;
						}

						$values = $mtrans->get('field_media_image')->getValue();
						if (empty($values)) {
							continue;
						}

						foreach ($values as $item) {
							$fid = isset($item['target_id']) ? (int) $item['target_id'] : 0;
							if (! $fid) {
								continue;
							}

							if ($skipProcessed && isset($processedIndex[$fid])) {
								$processedByAutoalt++;
								continue;
							}

							$key = $fid . '|' . $langcode;
							if (isset($seen[$key])) {
								continue;
							}

							$seen[$key] = TRUE;
						}
					}
				}
			}

			return new JsonResponse([
				'success' => TRUE,
				'processed_by_autoalt' => $processedByAutoalt,
			]);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('processedByAutoaltCount error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}

	public function totalImagesCount(Request $request)
	{
		try {
			$mq = \Drupal::entityQuery('media')->accessCheck(FALSE)->exists('field_media_image');
			$mids = $mq->execute();

			$pairs = [];

			if (!empty($mids)) {
				$medias = \Drupal\media\Entity\Media::loadMultiple($mids);

				foreach ($medias as $media) {

					$translations = $media->getTranslationLanguages();

					if (empty($translations)) {
						$base_lang = $media->language()->getId();
						$translations = [$base_lang => NULL];
					}

					foreach ($translations as $langcode => $langData) {

						if (is_object($langcode) && method_exists($langcode, 'getId')) {
							$langcode = $langcode->getId();
						}
						if (! $media->hasTranslation($langcode)) {
							continue;
						}
						$mtrans = $media->getTranslation($langcode);

						if (! $mtrans->hasField('field_media_image')) {
							continue;
						}

						$items = $mtrans->get('field_media_image');
						if ($items->isEmpty()) {
							continue;
						}

						foreach ($items as $field_item) {
							$fid = (int) $field_item->target_id;
							if (! $fid) {
								continue;
							}
							$pairs[] = ['fid' => $fid, 'langcode' => $langcode];
						}
					}
				}
			}

			return new JsonResponse([
				'success' => TRUE,
				'total_all' => count($pairs),
			]);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('ListFids error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}

	public function availCredits(Request $request)
	{
		try {
			$available_credits = 0;
			$config            = $this->configFactory->get('autoalt.settings');
			$apiKey            = $config->get('api_key') ?: '';

			if ($apiKey) {
				try {
					$resp = $this->httpClient->post('https://ahxdfj.autoalt.ai/api/autoalt-credit-count', [
						'headers' => [
							'Authorization' => 'Bearer ' . $apiKey,
							'Content-Type' => 'application/json',
							'Accept' => 'application/json',
						],
					]);
					if ($resp->getStatusCode() === 200) {
						$body = json_decode($resp->getBody()->getContents(), TRUE);
						$available_credits = (int) ($body['credit'] ?? 0);
					}
				} catch (\Exception $e) {
					\Drupal::logger('autoalt')->warning('Credit check failed: @e', ['@e' => $e->getMessage()]);
				}
			}

			return new JsonResponse([
				'success' => TRUE,
				'available_credits' => $available_credits,
			]);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('ListFids error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}

	public function historyList(Request $request)
	{
		try {
			$limit  = max(1, (int) $request->query->get('limit', 50)); 
			$offset = max(0, (int) $request->query->get('offset', 0));
			$fid = $request->query->get('fid', NULL);
			$lang = $request->query->get('language', NULL);
			$q = $request->query->get('q', NULL);

			$db = \Drupal::database();

			$conditions = [];
			if ($fid !== NULL && $fid !== '') {
				$conditions[] = ['field' => 'fid', 'value' => (int) $fid, 'operator' => '='];
			}
			if ($lang !== NULL && $lang !== '' && $lang !== 'all') {
				$conditions[] = ['field' => 'langcode', 'value' => $lang, 'operator' => '='];
			}
			if ($q !== NULL && $q !== '') {
				$conditions[] = ['field' => 'alt_text', 'value' => '%' . $db->escapeLike($q) . '%', 'operator' => 'LIKE'];
			}

			$count_query = $db->select('autoalt_history', 'h');
			$count_query->addExpression('COUNT(*)', 'cnt');
			foreach ($conditions as $c) {
				$count_query->condition($c['field'], $c['value'], $c['operator']);
			}
			$total = (int) $count_query->execute()->fetchField();

			$query = $db->select('autoalt_history', 'h')
				->fields('h', ['id', 'fid', 'mid', 'langcode', 'alt_text', 'processed_by', 'processed_on', 'source'])
				->orderBy('processed_on', 'DESC');

			foreach ($conditions as $c) {
				$query->condition($c['field'], $c['value'], $c['operator']);
			}

			if ($limit > 0) {
				$query->range($offset, $limit);
			}

			$results = $query->execute()->fetchAll(\PDO::FETCH_ASSOC);

			foreach ($results as &$row) {
				try {
					$file = \Drupal\file\Entity\File::load($row['fid']);
					if (!$file) {
						throw new \InvalidArgumentException('Invalid file ID.');
					}
					$file_uri = $file->getFileUri();
					$image_url = $this->fileUrlGenerator->generateAbsoluteString($file_uri);
					$row['thumbnail'] = $file_uri ? $image_url : '';
				} catch (\Exception $e) {
					$row['thumbnail'] = '';
				}
			}

			return new JsonResponse([
				'success' => TRUE,
				'rows' => $results,
				'total' => $total,
				'offset' => $offset,
				'limit' => $limit,
			]);
		} catch (\Exception $e) {
			\Drupal::logger('autoalt')->error('historyList error: @e', ['@e' => $e->getMessage()]);
			return new JsonResponse(['success' => FALSE, 'error' => $e->getMessage()], 500);
		}
	}

	public function historyPage()
	{
		$languages = \Drupal::languageManager()->getLanguages();

		$lang_list = [];
		foreach ($languages as $lang_code => $lang_obj) {
			$lang_list[$lang_code] = $lang_obj->getName();
		}
		asort($lang_list, SORT_NATURAL | SORT_FLAG_CASE);

		$csrf_token_service = \Drupal::service('csrf_token');
		$token = $csrf_token_service->get('');

		$build = [
			'#theme' => 'autoalt_history_page',
			'#language_list' => $lang_list,
			'#attached' => [
				'library' => [
					'autoalt/history_page',
				],
				'drupalSettings' => [
					'autoalt' => [
						'historyUrl' => '/api/autoalt/history',
						'saveAltUrl' => '/api/autoalt/save-alt',
						'defaultLimit' => 200,
						'defaultLanguage' => 'all',
						'csrfToken' => $token,
						'languageList' => $lang_list,
					],
				],
			],
			'#cache' => ['max-age' => 0],
		];

		return $build;
	}


	private function buildAndSendPluginPayload(?Request $request = NULL): array
	{
		$config = $this->configFactory->get('autoalt.settings');
		$apiKey = $config->get('api_key');

		if (empty($apiKey)) {
			throw new \RuntimeException('API key not configured');
		}

		$totalImages = (int) \Drupal::database()
			->select('media_field_data', 'm')
			->countQuery()
			->execute()
			->fetchField();

		$processedByPlugin = (int) \Drupal::database()
			->select('autoalt_history', 'h')
			->countQuery()
			->execute()
			->fetchField();

		$missingAlt = 0;

		$pluginInfo = \Drupal::service('extension.list.module')
			->getExtensionInfo('autoalt');

		$pluginVersion = $pluginInfo['version'] ?? 'unknown';

		$siteUrl = $request
			? $request->getSchemeAndHttpHost()
			: \Drupal::request()->getSchemeAndHttpHost();

		$pluginSettingData = [
			'prefix'                       => $config->get('prefix') ?: '',
			'suffix'                       => $config->get('suffix') ?: '',
			'ai_writing_style'             => $config->get('writing_style') ?: '',
			'customPromptFromUser'         => $config->get('custom_prompt') ?: '',
			'accessImgUrlPublic'           => (bool) $config->get('site_publicly_accessible'),
			'altTextMinLength'             => (int) $config->get('min_alt_length'),
			'altTextMaxLength'             => (int) $config->get('max_alt_length'),
			'setImageTitle'                => (bool) $config->get('auto_generate_image_name'),
			'imageExtensions'              => array_values((array) $config->get('filter_extensions')),
			'whenMediauploading'           => (bool) $config->get('auto_generate_on_upload'),
			'enableActiveSaleschannelLang' => true,
			'seo_keywords'                 => $config->get('seo_keywords') ?: '',
			'negative_keywords'            => $config->get('negative_keywords') ?: '',
		];

		$payload = [
			'total_images'        => $totalImages,
			'missing_alt'         => $missingAlt,
			'framework'           => 'drupal',
			'framework_version'   => \Drupal::VERSION,
			'plugin_version'      => $pluginVersion,
			'process_by_plugin'   => $processedByPlugin,
			'site_url'            => $siteUrl,
			'plugin_setting_data' => $pluginSettingData,
		];

		$this->httpClient->post(
			'https://ahxdfj.autoalt.ai/api/autoalt-get-data-from-plugin',
			[
				'headers' => [
					'Authorization' => 'Bearer ' . $apiKey,
					'Content-Type'  => 'application/json',
					'Accept'        => 'application/json',
				],
				'json' => $payload,
			]
		);

		return $payload;
	}

	public function sendPluginDataToRemote(Request $request)
	{
		try {
			$payload = $this->buildAndSendPluginPayload($request);

			return new JsonResponse([
				'success' => TRUE,
				'message' => 'Plugin data sent successfully',
				'data'    => $payload,
			]);
		} catch (\Throwable $e) {
			\Drupal::logger('autoalt')->error(
				'sendPluginDataToRemote error: @e',
				['@e' => $e->getMessage()]
			);

			return new JsonResponse([
				'success' => FALSE,
				'error'   => $e->getMessage(),
			], 500);
		}
	}


	public function collectAndSendPluginData(): void
	{
		try {
			$this->buildAndSendPluginPayload();
		} catch (\Throwable $e) {
			\Drupal::logger('autoalt')->warning(
				'Plugin sync failed: @msg',
				['@msg' => $e->getMessage()]
			);
		}
	}
}
