(function ($, Drupal, drupalSettings) {
	'use strict';

	$(document).ready(function () {
		const $start = $('#autoalt-start');
		const $cancel = $('#autoalt-cancel');
		const $progressWrapper = $('#autoalt-progress-wrapper');
		const $progressFill = $('#autoalt-progress-fill');
		const $progressCount = $('#autoalt-progress-count');
		const $progressCountPerc = $('#autoalt-progress-count-perc');
		const $progressTotal = $('#autoalt-progress-total');
		const $totalImages = $('#autoalt-total-images');
		const $credits = $('#autoalt-credits');

		let cancelRequested = false;

		function setGenerateButtonLabel(count) {
			const label = Drupal.t('Generate Alt Text : @n images', { '@n': count });
			$start.val(label);
			$start.data('count-to-process', count);
		}

		function getAvailableCredit() {
			return $.ajax({
				url: Drupal.url('api/autoalt/availcredit'),
				method: 'POST'
			}).then(function (resp) {
				if (resp && resp.success) {
					const availableCredits = resp.available_credits !== undefined ? resp.available_credits : ($credits.text() ? parseInt($credits.text(), 10) : 0);
					$credits.text(availableCredits);
					return parseInt(availableCredits, 10) || 0;
				}
				
				return $.Deferred().reject((resp && resp.error) ? resp.error : Drupal.t('API key missing or invalid.'));
			}, function (xhr) {
				const err = (xhr && xhr.responseJSON && xhr.responseJSON.error) ? xhr.responseJSON.error : Drupal.t('Could not fetch credits.');
				return $.Deferred().reject(err);
			});
		}

		getAvailableCredit().fail(function (err) {
			const messenger = new Drupal.Message();
			messenger.add(Drupal.t('AutoAlt: @msg', { '@msg': err }), { type: 'warning' });
		});

		function getTotalImagesCount() {
			$.ajax({
				url: Drupal.url('api/autoalt/totalImagesCount'),
				method: 'POST',
				success: function (resp) {
					if (resp && resp.success) {
						const totalAll = resp.total_all !== undefined ? resp.total_all : ($totalImages.text() ? parseInt($totalImages.text(), 10) : totalToProcess);

						$totalImages.text(totalAll);
						setGenerateButtonLabel(totalAll);
					}
				},
				error: function (xhr) {
				}
			});
		}


		function getProcessedByAutoaltCount() {
			$.ajax({
				url: Drupal.url('api/autoalt/processedByAutoaltCount'),
				method: 'POST',
				success: function (resp) {
					if (resp && resp.success) {
						if (resp.processed_by_autoalt > 0) {
							$('label[for="edit-skip-processed"]').append(
								' ( <b>' + resp.processed_by_autoalt + '</b> )'
							);
						}
					}
				},
				error: function (xhr) {
				}
			});
		}

		getProcessedByAutoaltCount();

		function refreshCountsFromServer(skipProcessed, cb) {
			$start.attr('disabled', 'true');

			$.ajax({
				url: Drupal.url('api/autoalt/list-fids'),
				method: 'POST',
				data: { skip_processed: skipProcessed ? 1 : 0 },
				success: function (resp) {

					if (resp && resp.success) {
						const totalToProcess = resp.totalAll !== undefined ? resp.totalAll : (resp.fids ? resp.fids.length : 0);
						setGenerateButtonLabel(totalToProcess);
						$start.removeAttr('disabled', 'disabled');

						if (typeof cb === 'function') cb(null, resp);
						return;
					}
					if (typeof cb === 'function') cb(resp || 'no-response');
				},
				error: function (xhr) {
					if (typeof cb === 'function') cb(xhr);
					$start.removeAttr('disabled', 'disabled');
				}
			});
		}

		getAvailableCredit();
		getTotalImagesCount();

		$('#autoalt-skip-processed').on('change', function () {
			const skipProcess = $(this).is(':checked');
			refreshCountsFromServer(skipProcess);
		});

		
		$start.on('click', function (e) {
			e.preventDefault();

			const skipProcessed = $('#autoalt-skip-processed').is(':checked') ? 1 : 0;
			const seoKeywords = $('#autoalt-seo-keywords').val() || '';
			const negativeKeywords = $('#autoalt-negative-keywords').val() || '';

			const countToProcess = $start.data('count-to-process') !== undefined ? parseInt($start.data('count-to-process'), 10) : parseInt($totalImages.text() || '0', 10);
			if (!countToProcess || countToProcess <= 0) {
				alert(Drupal.t('No images to process.'));
				return;
			}

			getAvailableCredit().done(function (availableCredits) {

				if (!availableCredits || availableCredits <= 0) {
					alert(Drupal.t('Insufficient credits (@c). Please add credits or check API key before bulk generation.', { '@c': availableCredits }));
					return;
				}

				if (!confirm(Drupal.t('Start generating Alt Text for @n images?', { '@n': countToProcess }))) {
					return;
				}

				$start.prop('disabled', true).text(Drupal.t('Preparing...'));

				$.ajax({
					url: Drupal.url('api/autoalt/list-fids'),
					method: 'POST',
					data: {
						skip_processed: skipProcessed
					},
					success: function (resp) {
						if (!resp || !resp.success) {
							alert(Drupal.t('Failed to retrieve images: @err', { '@err': resp && resp.error ? resp.error : 'Unknown' }));

							setGenerateButtonLabel(countToProcess || parseInt($totalImages.text() || '0', 10));
							$start.prop('disabled', false);
							return;
						}

						const fidsRaw = resp.fids || [];
						const grouped = resp.grouped || null;
						const normalized = [];

						if (Array.isArray(fidsRaw) && fidsRaw.length && typeof fidsRaw[0] === 'object') {
							fidsRaw.forEach(function (it) {
								const fid = parseInt(it.fid || it, 10);
								const lang = (it.langcode !== undefined) ? (it.langcode || '') : '';
								if (!isNaN(fid)) {
									normalized.push({ fid: fid, langcode: lang });
								}
							});
						} else if (Array.isArray(fidsRaw) && fidsRaw.length && (typeof fidsRaw[0] === 'number' || typeof fidsRaw[0] === 'string')) {

							fidsRaw.forEach(function (fidRaw) {
								const fid = parseInt(fidRaw, 10);
								if (!isNaN(fid)) {
									const fallbackLang = (drupalSettings.autoalt && drupalSettings.autoalt.lang) ? drupalSettings.autoalt.lang : '';
									normalized.push({ fid: fid, langcode: fallbackLang });
								}
							});
						} else if (grouped && typeof grouped === 'object') {
							Object.keys(grouped).forEach(function (fidKey) {
								const fid = parseInt(fidKey, 10);
								if (isNaN(fid)) return;
								const langs = grouped[fidKey] || [];
								if (Array.isArray(langs) && langs.length) {
									langs.forEach(function (lc) {
										normalized.push({ fid: fid, langcode: lc || '' });
									});
								} else {
									const fallbackLang = (drupalSettings.autoalt && drupalSettings.autoalt.lang) ? drupalSettings.autoalt.lang : '';
									normalized.push({ fid: fid, langcode: fallbackLang });
								}
							});
						}

						if (normalized.length === 0 && Array.isArray(fidsRaw) && fidsRaw.length) {
							fidsRaw.forEach(function (v) {
								const fid = parseInt((typeof v === 'object' ? v.fid : v), 10);
								if (!isNaN(fid)) {
									const fallbackLang = (drupalSettings.autoalt && drupalSettings.autoalt.lang) ? drupalSettings.autoalt.lang : '';
									normalized.push({ fid: fid, langcode: fallbackLang });
								}
							});
						}

						const queue = normalized;
						const total = queue.length;

						if (resp.total_all !== undefined) {
							$totalImages.text(resp.total_all);
						}

						if (total === 0) {
							alert(Drupal.t('No images to process.'));
							setGenerateButtonLabel(0);
							$start.prop('disabled', false);
							return;
						}

						$progressWrapper.show();
						$progressTotal.text(total);
						$progressCount.text(0);
						$progressCountPerc.text(0);
						$progressFill.css('width', '0%');
						cancelRequested = false;

						$cancel.off('click').on('click', function () {
							if (confirm(Drupal.t('Cancel bulk generation process?'))) {
								cancelRequested = true;
							}
						});

						
						const delayMs = 250;
						let processed = 0;

						function processNext() {
							if (cancelRequested) {
								$start.prop('disabled', false);
								setGenerateButtonLabel(total);
								alert(Drupal.t('Bulk generation cancelled. @n processed.', { '@n': processed }));
								return;
							}
							if (processed >= total) {
								$start.prop('disabled', false);
								setGenerateButtonLabel(total);
								alert(Drupal.t('Bulk generation finished. @n processed.', { '@n': processed }));
								return;
							}

							const entry = queue[processed];
							const fid = entry.fid;
							const langcode = entry.langcode || '';

							const payload = { fid: fid, language: langcode, 'seo_keywords': seoKeywords, 'negative_keywords': negativeKeywords };

							if (seoKeywords) payload.seo_keywords = seoKeywords;
							if (negativeKeywords) payload.negative_keywords = negativeKeywords;

							$.ajax({
								url: Drupal.url('api/autoalt/generate'),
								method: 'POST',
								data: payload,
								success: function (r) {

									if (r && r.success && r.altText) {
										$.ajax({
											url: Drupal.url('api/autoalt/save-alt'),
											method: 'POST',
											data: {
												fid: fid,
												alt: r.altText,
												language: langcode,
												imagename: r.imagename
											},
											success: function (saveResp) {
												if (saveResp && saveResp.success) {
												} else {
													console.warn('save-alt failed for fid', fid, 'lang', langcode, saveResp);
												}
											},
											error: function (xhr) {
												console.warn('save-alt request failed for fid', fid, 'lang', langcode, xhr);
											}
										});
									} else {
										console.warn('No altText returned for fid', fid, 'lang', langcode, r && r.error);
									}
								},
								error: function (xhr) {
									let msg = '';

									if (xhr.responseJSON) {
										if (xhr.responseJSON.error) msg = xhr.responseJSON.error;
										else if (xhr.responseJSON.message) msg = xhr.responseJSON.message;
										else if (xhr.responseJSON.raw) {
											try {
												const rawObj = JSON.parse(xhr.responseJSON.raw);
												msg = rawObj.message || rawObj.error || String(rawObj);
											} catch (e) {
												msg = xhr.responseJSON.raw; 
											}
										}
									}
									
									if (!msg && xhr.responseText) {
										try {
											const parsed = JSON.parse(xhr.responseText);
											msg = parsed.error || parsed.message || JSON.stringify(parsed);
										} catch (e) {
											msg = xhr.responseText; 
										}
									}
									
									if (!msg) msg = xhr.statusText || 'Unknown error';
									
									if (typeof msg === 'string' && msg.indexOf('Domain and key mismatch') !== -1) {
										alert('Domain and key mismatch. Please ensure both are correct.');
										location.reload(); 
										return; 
									}

									if (typeof msg === 'string' && msg.indexOf('Out of credit') !== -1) {
										alert('Out of credit.');
										location.reload(); 
										return; 
									}
									
								},
								complete: function () {
									processed++;
									const pct = Math.round((processed / total) * 100);

									$progressFill.css('width', pct + '%');
									$progressCountPerc.text(pct + '%');
									$progressCount.text(processed);

									setTimeout(processNext, delayMs);
								}
							});
						}

						processNext();
					},
					error: function () {
						alert(Drupal.t('Failed to fetch images to process.'));
						setGenerateButtonLabel(countToProcess || parseInt($totalImages.text() || '0', 10));
						$start.prop('disabled', false);
					}
				});
			}).fail(function (err) {
				alert(Drupal.t('Could not verify credits: @msg', { '@msg': err }));
			});
		});

	});

})(jQuery, Drupal, drupalSettings);
