(function ($, Drupal, drupalSettings, once) {
  'use strict';

  Drupal.behaviors.autoalt = {
    trackedImages: {},

    checkCreditsPromise: function () {

      return new Promise((resolve, reject) => {
        $.ajax({
          url: Drupal.url('api/autoalt/availcredit'),
          method: 'POST',
          success: function (resp) {
            if (!resp || !resp.success) {
              return reject((resp && resp.error) ? resp.error : Drupal.t('API key missing or invalid.'));
            }
            const credits = (resp.available_credits !== undefined) ? parseInt(resp.available_credits, 10) : 0;
            resolve(isNaN(credits) ? 0 : credits);
          },
          error: function (xhr) {
            let err = Drupal.t('Could not verify API key / credits. Please try again later.');
            if (xhr && xhr.responseJSON && xhr.responseJSON.error) err = xhr.responseJSON.error;
            reject(err);
          }
        });
      });
    },

    finishedWorking: function ($button) {
      const fid = $button.data('file-id');
      if (fid) {
        this.trackedImages[fid] = false;
      }
      $button.parent().find('.ajax-progress').remove();

      if (!(drupalSettings.autoalt && drupalSettings.autoalt.hide_button)) {
        $button.show();
      }

      $button
        .closest('.form-managed-file')
        .find("input[name$='[alt]']")
        .removeAttr('disabled');
    },

    generateForButton: function ($button) {
      if (!$button || !$button.length) return;

      const fid = $button.data('file-id');
      if (!fid) return;

      if (this.trackedImages[fid]) {
        return;
      }

      const messenger = new Drupal.Message();

      this.checkCreditsPromise().then((credits) => {
        if (!credits || credits <= 0) {
          messenger.add(Drupal.t('AutoAlt: insufficient credits (available: @c). Please add credits or check API key.', { '@c': credits }), { type: 'warning' });
          return;
        }
        
        this.trackedImages[fid] = true;

        const throbber = $(
          '<div class="ajax-progress autoalt-throbber--right">' +
          '<div class="ajax-progress__throbber">&nbsp;</div>' +
          '<div class="ajax-progress__message">' + Drupal.t('Generating Alt Text...') + '</div>' +
          '</div>'
        );

        const $container = $button.closest('.autoalt-wrapper').length ? $button.closest('.autoalt-wrapper') : $button.parent();
        $container.append(throbber);

        const $fileWrapper = $button.closest('.form-managed-file, .js-form-managed-file');
        $fileWrapper.find("input[name$='[alt]']").attr('disabled', 'disabled');
        $button.hide();

        let language = '';
        if (drupalSettings.autoalt.lang) {
          language = (drupalSettings.autoalt && drupalSettings.autoalt.lang) || '';
        } else {
          language = (drupalSettings.autoalt && drupalSettings.autoalt.language) || '';
        }

        $.ajax({
          url: Drupal.url('api/autoalt/generate'),
          method: 'POST',
          data: { fid: fid, language: language, saveHistory: 1 },

          success: (response) => {
            let altText = null;
            let imageName = null;

            if (response && response.success && response.altText) {
              altText = response.altText;
            } else if (response && response.alt_text) {
              altText = response.alt_text;
            }

            if (altText) {
              altText = altText.replace(/^"|"$/g, '');
              $fileWrapper.find("input[name$='[alt]']").val(altText);
            }

            if (response && response.imagename) {
              imageName = response.imagename;
            } else if (response && response.image_name) {
              imageName = response.image_name;
            } else if (response && response.name) {
              imageName = response.name;
            }

            if (imageName) {
              imageName = imageName.replace(/^"|"$/g, '');

              const $nameFieldSelectors = $fileWrapper.find(
                "input[id^='edit-name'], input[name^='name'], input[name$='[value]'], input[name*='[name]']"
              );

              if ($nameFieldSelectors.length) {
                const $target = $nameFieldSelectors.first();
                $target.val(imageName);

                $target.trigger('input').trigger('change');
              } else {

                const $closestForm = $fileWrapper.closest('form');
                const $globalNameField = $closestForm.find("input[id^='edit-name'], input[name^='name'], input[name$='[value]']");
                if ($globalNameField.length) {
                  const $target = $globalNameField.first();
                  $target.val(imageName);
                  $target.trigger('input').trigger('change');
                }
                else {
                  messenger.add(Drupal.t('AutoAlt: no matching name field found to set image name.'), { type: 'warning' });
                }
              }
            }

            this.finishedWorking($button);
          },

          error: (xhr) => {
            const messenger2 = new Drupal.Message();
            if (xhr.responseJSON && xhr.responseJSON.error) {
              messenger2.add('Error: ' + xhr.responseJSON.error, { type: 'warning' });
            } else {
              messenger2.add(Drupal.t('Could not generate alt text. Please try again later.'), { type: 'warning' });
            }

            this.finishedWorking($button);
          },
        });

      }).catch((errMsg) => {
        const messenger = new Drupal.Message();
        messenger.add(Drupal.t('AutoAlt: @msg', { '@msg': errMsg }), { type: 'warning' });
      });
    },

    attach: function (context) {
      const settings = drupalSettings.autoalt || {};
      const autoGenerate = settings.auto_generate_on_upload || settings.autoGenerateOnUpload || false;

      once('autoalt-button', '.autoalt-button, .ai-alt-text-generation', context).forEach((el) => {
        const $el = $(el);
        $el.on('click', function (e) {
          e.preventDefault();
          Drupal.behaviors.autoalt.generateForButton($el);
        });
      });

      once('autoalt-file', 'input[type="file"].js-form-file', context).forEach((el) => {
        $(el).on('change', function () {
          const $wrapper = $(this).closest('.js-form-managed-file');
          const $button = $wrapper.find('.autoalt-button, .ai-alt-text-generation');
          if ($button.length) {
            $button.hide();
          }
        });
      });

      if (autoGenerate) {
        $(context).find('.autoalt-button, .ai-alt-text-generation').each(function () {
          const $btn = $(this);
          const fid = $btn.data('file-id') || $btn.attr('data-file-id') || null;
          const $altField = $btn.closest('.form-managed-file').find("input[name$='[alt]']");

          if (fid && $altField.length && !$altField.val()) {
            if (!Drupal.behaviors.autoalt.trackedImages[fid]) {
              Drupal.behaviors.autoalt.generateForButton($btn);
            }
          }
        });
      }

      $(document).ajaxComplete((event, xhr, settings) => {
        if (!settings || !settings.url) return;

        const uploadDetected = settings.url.includes('/file/ajax/field_') || settings.url.includes('/media/ajax/');

        if (!uploadDetected) return;

        $('.js-form-managed-file').each(function () {
          const $wrapper = $(this);
          const $hiddenFid = $wrapper.find('input[type="hidden"][name$="[fids]"]');
          const $button = $wrapper.find('.autoalt-button, .ai-alt-text-generation');

          if ($hiddenFid.length && $hiddenFid.val() && $button.length) {
            const fid = $hiddenFid.val();
            $button.attr('data-file-id', fid).data('file-id', fid);
            $button.show();

            if (autoGenerate) {
              const $altField = $wrapper.find("input[name$='[alt]']");
              if ($altField.length && !$altField.val()) {
                if (!Drupal.behaviors.autoalt.trackedImages[fid]) {
                  Drupal.behaviors.autoalt.generateForButton($button);
                }
              }
            }
          }
        });
      });
    },
  };
})(jQuery, Drupal, drupalSettings, once);
