<?php

namespace Drupal\autoalt\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

class AutoaltBulkAltForm extends FormBase
{

	/**
	 * {@inheritdoc}
	 */
	public function getFormId()
	{
		return 'autoalt_bulk_alt_form';
	}

	/**
	 * {@inheritdoc}
	 */
	public function buildForm(array $form, FormStateInterface $form_state)
	{
		$config       = $this->config('autoalt.settings');
		$total_images = 0;
		$missing      = 0;
		$available_credits = 0;

		$form['dashboard_cards'] = [
			'#type' => 'markup',
			'#markup' => '
				<div class="autoalt-dashboard">
					<div class="autoalt-cards">
						<div class="autoalt-card"><div class="label">Total Images</div><div class="value" id="autoalt-total-images">' . $total_images . '</div></div>
						
						<div class="autoalt-card"><div class="label">Available Credits</div><div class="value" id="autoalt-credits">' . $available_credits . '</div></div>
					</div>
				</div>',
		];
		
		$form['keywords_container'] = [
			'#type' => 'container',
			'#attributes' => [
				'class' => ['autoalt-keywords-row'],
			],
		];

		$form['keywords_container']['seo_keywords'] = [
			'#type' => 'textfield',
			'#title' => $this->t('SEO Keywords (optional)'),
			'#description' => $this->t('Comma-separated SEO keywords to include in alt text.'),
			'#default_value' => '',
			'#attributes' => [
				'id' => 'autoalt-seo-keywords',
				'placeholder' => 'Keyword1, Keyword2',
			],
		];

		$form['keywords_container']['negative_keywords'] = [
			'#type' => 'textfield',
			'#title' => $this->t('Negative Keywords (optional)'),
			'#description' => $this->t('Comma-separated words to avoid in alt text.'),
			'#default_value' => '',
			'#attributes' => [
				'id' => 'autoalt-negative-keywords',
				'placeholder' => 'Negative1, Negative2',
			],
		];

		$form['options'] = [
			'#type' => 'container',
			'#attributes' => ['class' => ['autoalt-options-wrapper']],
		];

		$form['options']['skip_processed'] = [
			'#type' => 'checkbox',
			'#title' => $this->t('Skip Images Already Processed by AutoAlt.ai '),
			'#default_value' => FALSE,
			'#attributes' => ['id' => 'autoalt-skip-processed'],
		];

		$form['actions'] = [
			'#type' => 'container',
			'#attributes' => ['class' => ['autoalt-actions-wrapper']],
		];

		$form['actions']['start'] = [
			'#type' => 'button',
			'#value' => $this->t('Generate Alt Text : @n images', ['@n' => $missing]),
			'#attributes' => [
				'id' => 'autoalt-start',
				'class' => ['button', 'button--primary'],
			],
		];

		$form['actions']['cancel'] = [
			'#type' => 'button',
			'#value' => $this->t('Cancel'),
			'#attributes' => [
				'id' => 'autoalt-cancel',
				'class' => ['button'],
				'style' => 'margin-left:6px;',
			],
		];

		$form['progress'] = [
			'#type' => 'container',
			'#attributes' => ['id' => 'autoalt-progress-wrapper', 'style' => 'display:none1;'],
			'bar' => [
				'#type' => 'markup',
				'#markup' => '<div class="autoalt-progress-info"> <span id="autoalt-progress-count">0</span> / <span id="autoalt-progress-total">0</span> Images processed</div><div class="autoalt-progress-bar"><div id="autoalt-progress-fill"></div></div><div class="autoalt-progress-info"> Process completed <span id="autoalt-progress-count-perc">0</span></div>',
			],
		];

		// Attach the dashboard library (CSS + JS).
		$form['#attached']['library'][] = 'autoalt/bulk_dashboard';

		return $form;
	}

	/**
	 * {@inheritdoc}
	 */
	public function submitForm(array &$form, FormStateInterface $form_state)
	{
		$rows  = $form_state->getValue('table');
		$saved = 0;
		foreach ($rows as $row) {
			if (!empty($row['alt'])) {
				$saved++;
			}
		}
		$this->messenger()->addStatus($this->t('@count alt texts saved.', ['@count' => $saved]));
	}

	/**
	 * Simple scanner: returns an array of items with file entity and parent entity.
	 * Replace with the robust scanner from the competitor code.
	 */
	protected function getEntitiesAndFields($limit = 50)
	{
		$items = [];

		$query = \Drupal::entityQuery('file')
			->condition('filemime', 'image/%', 'LIKE')
			->range(0, $limit);

		$query->accessCheck(FALSE);

		$fids = $query->execute();

		foreach ($fids as $fid) {
			$file = \Drupal\file\Entity\File::load($fid);
			if ($file) {
				$items[] = [
					'file' => $file,
					'base_entity' => $file,
					'unique_id' => md5($fid . time() . rand()),
				];
			}
		}
		return $items;
	}
}
