(function ($, Drupal, drupalSettings) {
  'use strict';

  $(document).ready(function () {
    
    const $tableBody  = $('#autoalt-history-table tbody');
    const $lang       = $('#autoalt-history-lang');
    const $search     = $('#autoalt-history-search');
    const $pagination = $('#autoalt-history-pagination');

    let limit  = 10;
    let offset = 0;
    let total  = 0;

    function renderPagination() {
      if (!$pagination.length) return;

      const currentPage = Math.floor(offset / limit) + 1;
      const totalPages = Math.max(1, Math.ceil(total / limit));

      const html = '<div class="autoalt-pagination-controls">' +
        '<button id="autoalt-page-prev"' + (offset === 0 ? ' disabled' : '') + '>Previous</button>' +
       
        '<button id="autoalt-page-next"' + (offset + limit >= total ? ' disabled' : '') + '>Next</button>' +
         '<span class="autoalt-page-info"> Page ' + currentPage + ' / ' + totalPages + ' (' + total + ' Records)</span>' +
        '</div>';

      $pagination.html(html);
    }

    function fetchHistory() {
      const langVal = $lang.length ? $lang.val() : 'all';
      const q = $search.val() || '';
      $.ajax({
        url: Drupal.url((drupalSettings && drupalSettings.autoalt && drupalSettings.autoalt.historyUrl) ? drupalSettings.autoalt.historyUrl : '/api/autoalt/history'),
        method: 'GET',
        data: { language: langVal, q: q, limit: limit, offset: offset },
        success: function (resp) {
          $tableBody.empty();
          total = (resp && resp.total) ? parseInt(resp.total, 10) : 0;

          if (resp && resp.success && Array.isArray(resp.rows) && resp.rows.length) {
            const langMap = (drupalSettings && drupalSettings.autoalt && drupalSettings.autoalt.languageList) ? drupalSettings.autoalt.languageList : {};

            resp.rows.forEach(function (row) {
              const thumb = row.thumbnail ? '<a href="'+ row.thumbnail +'" target="_blank"><img src="' + row.thumbnail + '" alt=""  /></a>' : '';
              const processedOn = row.processed_on ? new Date(row.processed_on * 1000).toLocaleString() : '';
              const altTd = '<textarea class="autoalt-history-alt" data-id="' + row.id + '">' + (row.alt_text || '') + '</textarea>';
              const actions = '<button type="button" class="autoalt-update-btn" data-fid="' + row.fid + '" data-lang="' + (row.langcode || '') + '" data-id="' + row.id + '">Update</button>';
              const langName = (row.langcode && langMap[row.langcode]) ? langMap[row.langcode] : (row.langcode || '');
              const hid = row.id;
              const tr = '<tr>' +
                '<td>' + hid + '</td>' +
                '<td>' + thumb + '</td>' +
                '<td>' + altTd + '</td>' +
                '<td>' + actions + '</td>' +
                '<td>' + langName + '</td>' +
                '<td>' + processedOn + '</td>' +
                
                '</tr>';
              $tableBody.append(tr);
            });
          } else {
            $tableBody.append('<tr><td colspan="5">No history found</td></tr>');
          }

          renderPagination();
        },
        error: function () {
          $tableBody.html('<tr><td colspan="5">Error fetching history</td></tr>');
          total = 0;
          renderPagination();
        }
      });
    }

    fetchHistory();

   
    $('#autoalt-history-refresh').on('click', function () {
      offset = 0;
      fetchHistory();
    });

    
    $(document).on('click', '#autoalt-page-prev', function (e) {
      if (offset === 0) return;
      offset = Math.max(0, offset - limit);
      fetchHistory();
    });
    $(document).on('click', '#autoalt-page-next', function (e) {
      if (offset + limit >= total) return;
      offset = offset + limit;
      fetchHistory();
    });

    
    $lang.on('change', function () {
      offset = 0;
      fetchHistory();
    });
    $search.on('keypress', function (e) {
      if (e.which === 13) {
        offset = 0;
        fetchHistory();
      }
    });

    
    $(document).on('click', '.autoalt-update-btn', function (e) {
      e.preventDefault();

      const $btn = $(this);
      const fid  = $btn.data('fid');
      const lang = $btn.data('lang') || '';
      const id   = $btn.data('id'); 
      
      const $ta = $btn.closest('tr').find('textarea.autoalt-history-alt');
      const alt = $ta.val() || '';

      if (!alt) {
        alert(Drupal.t('Alt text is empty.'));
        return;
      }
      if (!confirm(Drupal.t('Update alt text for this image?'))) {
        return;
      }

      $btn.prop('disabled', true).text(Drupal.t('Updating...'));

      $.ajax({
        url: Drupal.url('api/autoalt/save-alt'),
        method: 'POST',
        data: { fid: fid, alt: alt, language: lang, update_id: id },
        beforeSend: function (xhr) {
          if (drupalSettings && drupalSettings.autoalt && drupalSettings.autoalt.csrfToken) {
            xhr.setRequestHeader('X-CSRF-Token', drupalSettings.autoalt.csrfToken);
          }
        },
        success: function (resp) {
          if (resp && resp.success) {
            alert(Drupal.t('Alt text updated.'));
            fetchHistory();
          } else {
            alert(Drupal.t('Failed to update: @err', { '@err': resp && resp.error ? resp.error : 'Unknown' }));
          }
        },
        error: function () {
          alert(Drupal.t('Error while updating.'));
        },
        complete: function () {
          $btn.prop('disabled', false).text(Drupal.t('Update'));
        }
      });
    });

  });
})(jQuery, Drupal, drupalSettings);
