<?php

namespace Drupal\autoalt\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\autoalt\Controller\AutoaltController;
use GuzzleHttp\Exception\ClientException;

class AutoaltGeneratorSettingsForm extends ConfigFormBase
{

  protected const WRITING_STYLE = [
    '' => 'Use account default',
    'neutral' => 'Neutral — Clear, balanced, and objective',
    'friendly' => 'Friendly — Warm, approachable, and informal',
    'professional' => 'Professional — Polished and business-appropriate',
    'casual' => 'Casual — Laid-back and conversational',
    'witty' => 'Witty — Clever, playful, with light humor',
    'confident' => 'Confident — Assertive and bold',
    'empathetic' => 'Empathetic — Sensitive and understanding',
    'inspiring' => 'Inspiring — Motivational and uplifting',
    'technical' => 'Technical — Precise, jargon-friendly, and expert-level',
    'minimalist' => 'Minimalist — Simple, clean, and concise',
    'luxury' => 'Luxury — Elegant and premium-sounding',
    'youthful' => 'Youthful — Trendy, fresh, and informal',
    'quirky' => 'Quirky — Fun, offbeat, and creative',
    'sales_oriented' => 'Sales-Oriented — Persuasive and call-to-action driven',
    'informative' => 'Informative — Educational and fact-based',
    'formal' => 'Formal — Structured, respectful, and traditional',
    'authoritative' => 'Authoritative — Expert and trustworthy',
    'seo_optimized' => 'SEO-Optimized — Keyword-rich and search-focused',
  ];

  public function getFormId()
  {
    return 'autoalt_settings';
  }

  protected function getEditableConfigNames()
  {
    return ['autoalt.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config      = $this->config('autoalt.settings');
    $apiKey      = $config->get('api_key');
    $isConnected = !empty($apiKey);

    $form['auth_grid'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['autoalt-auth-grid']],
    ];

    $form['auth_grid']['header'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['autoalt-auth-header']],
    ];

    $form['auth_grid']['header']['title'] = [
      '#markup' => '<h2 class="autoalt-auth-title">Welcome to AutoAlt.ai</h2>',
    ];

    $form['auth_grid']['header']['subtitle'] = [
      '#markup' => '<p class="autoalt-auth-subtitle">
            Generate AI alt text faster with your AutoAlt.ai account.
          </p>',
    ];

    $form['auth_grid']['columns'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => array_filter([
          'autoalt-auth-grid-columns',
          $isConnected ? 'autoalt-grid-single' : NULL,
        ]),
      ],
    ];

    $form['auth_grid']['columns']['api_key_card'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['autoalt-card']],
    ];

    $form['auth_grid']['columns']['api_key_card']['header'] = [
      '#markup' => '
        <div class="autoalt-card-header">
          <div class="autoalt-card-icon">🔑</div>
          <div>
            <h3 class="autoalt-card-title">Connect API Key</h3>
            <p class="autoalt-card-subtitle">Already have an API key?</p>
          </div>
        </div>',
    ];

    $form['auth_grid']['columns']['api_key_card']['api_key'] = [
      '#type' => 'textfield',
      '#default_value' => $apiKey,
      '#placeholder' => $this->t('Paste your AutoAlt.ai API key'),
      '#attributes' => ['class' => ['autoalt-input']],
    ];

    if ($apiKey) {
      try {
        $client = new Client();
        $response = $client->post('https://ahxdfj.autoalt.ai/api/autoalt-credit-count', [
          'headers' => [
            'Authorization' => 'Bearer ' . $apiKey,
            'Accept' => 'application/json',
          ],
        ]);

        $data = json_decode($response->getBody(), TRUE);

        $form['auth_grid']['columns']['api_key_card']['autoalt_credits_info'] = [
          '#type' => 'markup',
          '#markup' => '<div class="credit-info-box">' .
            $this->t('Your current credit balance is @c / @t', [
              '@c' => $data['credit'] ?? 0,
              '@t' => $data['total_credit'] ?? 0,
            ]) .
            '</div>',
        ];
      } catch (GuzzleException $e) {
        \Drupal::logger('autoalt')->error($e->getMessage());
      }
    }

    $form['auth_grid']['columns']['api_key_card']['save_api'] = [
      '#type' => 'submit',
      '#value' => $isConnected ? $this->t('Connected') : $this->t('Connect & Save'),
      '#attributes' => [
        'class' => [
          'autoalt-btn',
          $isConnected ? 'autoalt-btn-success' : 'autoalt-btn-primary'
        ],
        $isConnected ? 'disabled' : NULL,
      ],
      '#disabled' => $isConnected,
    ];

    if ($isConnected) {
      $form['auth_grid']['columns']['api_key_card']['clear_api'] = [
        '#type' => 'button',
        '#value' => $this->t('Clear API key'),
        '#attributes' => [
          'class' => ['autoalt-btn', 'autoalt-btn-danger', 'autoalt-clear-api'],
        ],
        '#ajax' => [
          'callback' => '::clearApiKeyAjax',
          'wrapper' => 'autoalt-register-box',
        ],
      ];
    }

    if (!$isConnected) {
      $form['auth_grid']['columns']['register_box'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['autoalt-card']],
        '#prefix' => '<div id="autoalt-register-box">',
        '#suffix' => '</div>',
      ];

      $form['auth_grid']['columns']['register_box']['header'] = [
        '#markup' => '
          <div class="autoalt-card-header">
            <div class="autoalt-card-icon">👤</div>
            <div>
              <h3 class="autoalt-card-title">Login / Register</h3>
              <p class="autoalt-card-subtitle">New user? Get your API key</p>
            </div>
          </div>',
      ];

      $form['auth_grid']['columns']['register_box']['register_email'] = [
        '#type' => 'email',
        '#placeholder' => $this->t('Enter Email ID'),
        '#attributes' => ['class' => ['autoalt-input']],
      ];

      $form['auth_grid']['columns']['register_box']['send_otp'] = [
        '#type' => 'button',
        '#value' => $this->t('Send Verification Code'),
        '#attributes' => ['class' => ['autoalt-btn', 'autoalt-btn-primary']],
        '#ajax' => [
          'callback' => '::sendOtpAjax',
          'wrapper' => 'autoalt-register-box',
        ],
      ];

      $form['auth_grid']['columns']['register_box']['otp_wrapper'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['autoalt-otp-wrapper'],
          'style' => 'display:none;',
        ],
      ];

      $form['auth_grid']['columns']['register_box']['otp_wrapper']['otp_inputs'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['autoalt-otp-inputs']],
      ];

      for ($i = 1; $i <= 6; $i++) {
        $form['auth_grid']['columns']['register_box']['otp_wrapper']['otp_inputs']["otp_$i"] = [
          '#type' => 'textfield',
          '#maxlength' => 1,
          '#attributes' => [
            'class' => ['autoalt-otp-box'],
            'inputmode' => 'numeric',
            'autocomplete' => 'one-time-code',
          ],
        ];
      }

      $form['auth_grid']['columns']['register_box']['otp_wrapper']['verify_otp'] = [
        '#type' => 'button',
        '#value' => $this->t('Verify OTP'),
        '#attributes' => ['class' => ['autoalt-btn', 'autoalt-btn-primary']],
        '#ajax' => [
          'callback' => '::verifyOtpAjax',
          'wrapper' => 'autoalt-register-box',
        ],
      ];

      $form['auth_grid']['columns']['register_box']['otp_wrapper']['otp_timer'] = [
        '#markup' => '<div class="autoalt-otp-timer">
                    OTP expires in <span>05:00</span>
                  </div>',
      ];

      $form['auth_grid']['columns']['register_box']['otp_wrapper']['resend_otp'] = [
        '#type' => 'button',
        '#value' => $this->t('Resend verification code'),
        '#attributes' => [
          'class' => ['autoalt-btn', 'autoalt-btn-secondary', 'autoalt-resend-otp'],
          'style' => 'display:none;',
        ],
        '#ajax' => [
          'callback' => '::sendOtpAjax',
          'wrapper' => 'autoalt-register-box',
        ],
      ];
    }

    $form['writing_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Writing Style'),
      '#description' => $this->t('Choose the tone for generating the alt text.'),
      '#options' => self::WRITING_STYLE,
      '#default_value' => $config->get('writing_style') ?: '',
    ];

    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix for Alt Text'),
      '#description' => $this->t('Optional prefix to prepend to generated alt text.'),
      '#default_value' => $config->get('prefix') ?: '',
    ];

    $form['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix for Alt Text'),
      '#description' => $this->t('Optional suffix to append to generated alt text.'),
      '#default_value' => $config->get('suffix') ?: '',
    ];

    $form['filter_extensions'] = [
      '#type' => 'select',
      '#title' => $this->t('Filter by Image Extension'),
      '#description' => $this->t('Only generate alt text for selected image extensions.'),
      '#options' => [
        'jpg'  => 'jpg',
        'jpeg' => 'jpeg',
        'png'  => 'png',
        'gif'  => 'gif',
        'webp' => 'webp',
        'svg'  => 'svg',
      ],
      '#default_value' => $config->get('filter_extensions') ?: [],
      '#multiple' => TRUE,
      '#size' => 6,
    ];

    $form['auto_generate_image_name'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatic set image name when generate alt text'),
      '#default_value' => $config->get('auto_generate_image_name') ?? true,
    ];

    $form['auto_generate_on_upload'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-Generate on Image Upload'),
      '#default_value' => $config->get('auto_generate_on_upload') ?? true,
    ];

    $form['site_publicly_accessible'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('This site is publicly accessible on the internet'),
      '#default_value' => $config->get('site_publicly_accessible') ?? true,
      '#suffix' => "<div id='autoalt-cloudflare-note'>
      <div id='autoalt-cloudflare-note-heading'>
        Notice for Cloudflare or .htpasswd Protected Sites
      </div>
      <p>
        If your website is using <strong>Cloudflare’s “Under Attack Mode”</strong> or is 
        <strong>protected with an .htpasswd,</strong> please  <mark>Unchecked</mark> the checkbox
        <strong>“This site is publicly accessible on the internet.”</strong>
      </p>
    </div>",
    ];

    $form['seo_keywords'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SEO Keywords'),
      '#description' => $this->t('Comma-separated keywords to bias the generated alt text toward SEO.'),
      '#default_value' => $config->get('seo_keywords') ?: '',
    ];

    $form['negative_keywords'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Negative Keywords'),
      '#description' => $this->t('Comma-separated words to avoid in generated alt text.'),
      '#default_value' => $config->get('negative_keywords') ?: '',
    ];

    $form['custom_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt'),
      '#description' => $this->t('A custom prompt to send to the AI for generating alt text.'),
      '#default_value' => $config->get('custom_prompt') ?: '',
    ];

    $form['min_alt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum Alt Text Length'),
      '#default_value' => $config->get('min_alt_length') ?: 100,
      '#description' => $this->t('Minimum number of characters for alt text ( 50 - 100 )'),
      '#min' => 50,
      '#max' => 100,
    ];

    $form['max_alt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Alt Text Length'),
      '#default_value' => $config->get('max_alt_length') ?: 125,
      '#description' => $this->t('Maximum number of characters for alt text ( 100 - 250 )'),
      '#min' => 100,
      '#max' => 250,
    ];

    $form['#attached']['library'][] = 'autoalt/admin-settings';

    return parent::buildForm($form, $form_state);
  }

  public function sendOtpAjax(array &$form, FormStateInterface $form_state)
  {
    try {
      (new Client())->post(
        'https://ahxdfj.autoalt.ai/api/autoalt-send-otp',
        [
          'json' => [
            'email'  => $form_state->getValue('register_email'),
            'domain' => \Drupal::request()->getHost(),
          ],
          'timeout' => 10,
        ]
      );

      \Drupal::messenger()->addStatus(
        $this->t('Verification code sent to your email.')
      );
    } catch (ClientException $e) {
      $responseBody = $e->getResponse()->getBody()->getContents();
      $data = json_decode($responseBody, TRUE);

      if (!empty($data['error']) && $data['error'] === 'domain_exist') {
        $email = $data['domains']['user_email'] ?? '';
        \Drupal::messenger()->addError(
          $this->t(
            'This domain is already registered with the email %email. Please use the API key associated with this email.',
            ['%email' => $email]
          )
        );
      } else {
        \Drupal::messenger()->addError(
          $this->t('Failed to send OTP. Please try again.')
        );
      }
    } catch (\Exception $e) {
      \Drupal::logger('autoalt')->error($e->getMessage());

      \Drupal::messenger()->addError(
        $this->t('Something went wrong. Please try again later.')
      );
    }

    return $form['auth_grid']['columns']['register_box'];
  }

  public function verifyOtpAjax(array &$form, FormStateInterface $form_state)
  {
    $response = new AjaxResponse();

    $otp = '';
    for ($i = 1; $i <= 6; $i++) {
      $otp .= $form_state->getValue("otp_$i");
    }

    try {
      $api_response = (new Client())->post(
        'https://ahxdfj.autoalt.ai/api/autoalt-verify-otp',
        [
          'json' => [
            'email' => $form_state->getValue('register_email'),
            'otp' => $otp,
            'domain' => \Drupal::request()->getHost(),
          ],
          'timeout' => 10,
        ]
      );

      $data = json_decode($api_response->getBody(), TRUE);

      if (!empty($data['api_key'])) {

        $this->config('autoalt.settings')
          ->set('api_key', $data['api_key'])
          ->save();

        $response->addCommand(new HtmlCommand(
          '.messages',
          '<div class="messages messages--status">OTP verified successfully. API key connected.</div>'
        ));

        $response->addCommand(new InvokeCommand(
          '#autoalt-register-box .autoalt-otp-wrapper',
          'hide'
        ));

        $response->addCommand(new InvokeCommand(
          'input[name="api_key"]',
          'val',
          [$data['api_key']]
        ));

        $response->addCommand(new InvokeCommand(
          'body',
          'append',
          ['<script>window.location.reload();</script>']
        ));
      } else {
        $response->addCommand(new HtmlCommand(
          '.messages',
          '<div class="messages messages--error">Invalid OTP.</div>'
        ));
      }
    } catch (\Exception $e) {
      $response->addCommand(new HtmlCommand(
        '.messages',
        '<div class="messages messages--error">OTP verification failed. Please try again.</div>'
      ));
    }

    return $response;
  }

  public function clearApiKeyAjax(array &$form, FormStateInterface $form_state)
  {
    $response = new AjaxResponse();

    $this->config('autoalt.settings')
      ->set('api_key', '')
      ->save();

    $response->addCommand(new InvokeCommand(
      'input[name="api_key"]',
      'val',
      ['']
    ));

    $response->addCommand(new InvokeCommand(
      'body',
      'append',
      ['<script>window.location.reload();</script>']
    ));

    return $response;
  }

  private function validateApiKey(string $apiKey): bool
  {
    try {
      $client = new Client([
        'timeout' => 10,
      ]);

      $response = $client->post(
        'https://ahxdfj.autoalt.ai/api/autoalt-credit-count',
        [
          'headers' => [
            'Authorization' => 'Bearer ' . $apiKey,
            'Accept' => 'application/json',
          ],
        ]
      );

      $data = json_decode($response->getBody(), TRUE);

      return isset($data['credit'], $data['total_credit']);
    } catch (\Exception $e) {
      \Drupal::logger('autoalt')->warning('Invalid API key: @msg', [
        '@msg' => $e->getMessage(),
      ]);
      return false;
    }
  }

  private function syncPluginDataDirectly(): void
  {
    try {
      $controller = \Drupal::service('class_resolver')
        ->getInstanceFromDefinition(AutoaltController::class);
      $controller->collectAndSendPluginData();
    } catch (\Throwable $e) {
      \Drupal::logger('autoalt')->warning(
        'Direct plugin sync failed: @msg',
        ['@msg' => $e->getMessage()]
      );
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $apiKey = trim($form_state->getValue('api_key'));

    if (!$this->validateApiKey($apiKey)) {
      \Drupal::messenger()->addError(
        $this->t('Invalid API key. Please check and try again.')
      );
      return;
    }

    $this->config('autoalt.settings')
      ->set('api_key', $apiKey)
      ->set('writing_style', $form_state->getValue('writing_style'))
      ->set('prefix', $form_state->getValue('prefix'))
      ->set('suffix', $form_state->getValue('suffix'))
      ->set('filter_extensions', array_filter((array) $form_state->getValue('filter_extensions')))
      ->set('auto_generate_on_upload', (bool) $form_state->getValue('auto_generate_on_upload'))
      ->set('auto_generate_image_name', (bool) $form_state->getValue('auto_generate_image_name'))
      ->set('site_publicly_accessible', (bool) $form_state->getValue('site_publicly_accessible'))
      ->set('seo_keywords', $form_state->getValue('seo_keywords'))
      ->set('negative_keywords', $form_state->getValue('negative_keywords'))
      ->set('custom_prompt', $form_state->getValue('custom_prompt'))
      ->set('min_alt_length', (int) $form_state->getValue('min_alt_length'))
      ->set('max_alt_length', (int) $form_state->getValue('max_alt_length'))
      ->save();

    if (!empty($apiKey)) {
      $this->syncPluginDataDirectly();
    }
  }
}
